package net.lulihu.http.builder;

import net.lulihu.ObjectKit.StrKit;
import net.lulihu.http.MIME;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;

import java.io.File;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Http Post多部分请求构建
 */
public class HttpPostMultipartBuilder extends HttpBuilder {

    HttpPostMultipartBuilder(String url) {
        super(url);
        this.builder = new MultipartBody.Builder();
    }

    private MultipartBody.Builder builder;

    /**
     * 添加多部分请求参数
     *
     * @param key   键
     * @param value 值
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    public HttpPostMultipartBuilder addParam(String key, String value) {
        builder.addFormDataPart(key, value);
        return this;
    }

    /**
     * 添加多部分请求参数
     *
     * @param key   键
     * @param files 文件集
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    public HttpPostMultipartBuilder addParam(String key, List<File> files) {
        for (File file : files) {
            addParam(key, file);
        }
        return this;
    }


    /**
     * 添加多部分请求参数
     *
     * @param key  键
     * @param file 文件
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    public HttpPostMultipartBuilder addParam(String key, File file) {
        String name = file.getName();
        String sux = StrKit.subSuf(name, name.lastIndexOf("."));
        MediaType mediaType = MediaType.parse(MIME.getMimeType(sux));
        builder.addFormDataPart(key, name, RequestBody.create(mediaType, file));
        return this;
    }

    /**
     * 设置请求头信息
     *
     * @param header 头部信息
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    public HttpPostMultipartBuilder setHeader(Map<String, String> header) {
        super.setHeader(header);
        return this;
    }

    /**
     * 添加请求头信息
     *
     * @param key   键
     * @param value 值
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    public HttpPostMultipartBuilder addHeader(String key, String value) {
        super.addHeader(key, value);
        return this;
    }

    /**
     * 设置连接超时时间
     *
     * @param timeout 超时时间
     * @param unit    时间单位
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    public HttpPostMultipartBuilder setConnectTimeout(long timeout, TimeUnit unit) {
        super.setConnectTimeout(timeout, unit);
        return this;
    }

    /**
     * 设置读取超时时间
     *
     * @param timeout 超时时间
     * @param unit    时间单位
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    public HttpPostMultipartBuilder setReadTimeout(long timeout, TimeUnit unit) {
        super.setReadTimeout(timeout, unit);
        return this;
    }

    /**
     * 添加请求拦截器
     *
     * @param requestInterceptor 请求拦截器
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    @Override
    public HttpPostMultipartBuilder addRequestInterceptor(RequestInterceptor requestInterceptor) {
        super.addRequestInterceptor(requestInterceptor);
        return this;
    }

    /**
     * 添加响应拦截器
     *
     * @param responseInterceptor 响应拦截器
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    @Override
    public HttpPostMultipartBuilder addResponseInterceptor(ResponseInterceptor responseInterceptor) {
        super.addResponseInterceptor(responseInterceptor);
        return this;
    }

    /**
     * 添加网络层拦截器
     *
     * @param interceptor 网络层拦截器
     * @return {@linkplain HttpPostMultipartBuilder}
     */
    @Override
    public HttpPostMultipartBuilder addNetInterceptor(Interceptor interceptor) {
        super.addNetInterceptor(interceptor);
        return this;
    }

    /**
     * 发送Post载荷请求
     *
     * @return {@linkplain HttpPostMultipartBuilder}
     * @throws HttpRequestException 请求出现例外
     */
    public HttpPostMultipartBuilder send() throws HttpRequestException {
        // 请求地址
        requestBuilder.url(url);
        // 构建请求主体参数
        requestBuilder.post(builder.build());
        // 发送请求
        super.sendRequest();
        return this;
    }

}
