package net.linksfield.cube.partnersdk.sdk.proxy;

import lombok.Getter;
import net.linksfield.cube.partnersdk.domain.BaseRequest;
import net.linksfield.cube.partnersdk.domain.Enums;
import net.linksfield.cube.partnersdk.domain.SignatureBuilder;
import net.linksfield.cube.partnersdk.event.Events;
import net.linksfield.cube.partnersdk.event.events.DomainSignatureEvent;
import net.linksfield.cube.partnersdk.rest.*;
import net.linksfield.cube.partnersdk.sdk.ServicesContainer;
import net.linksfield.cube.partnersdk.sdk.proxy.extend.IModuleExecutor;
import net.linksfield.cube.partnersdk.sdk.proxy.extend.ModuleExecute;
import net.linksfield.cube.partnersdk.utils.ReflectUtils;
import org.apache.http.HttpStatus;

import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

/**
 * @ClassName AbstractModuleImplementProxy
 * @Description SDK Module 实现的动态代理
 * @Author James.hu
 * @Date 2023/3/15
 **/
public abstract class AbstractModuleImplementProxy implements InvocationHandler {

    @Getter
    private Enums.SIGN_TYPE signType;

    /**
     * 动态代理了Module的接口实现
     * 假定第一个为BaseRequest类型, 并且只使用第一个参数调用 Http实现的发送方法
     */
    @Override
    public Object invoke(Object proxy, Method method, Object[] args) {
        // 假定第一个参数必须为BaseRequest类型
        BaseRequest domain = (BaseRequest) args[0];
        
        // 判断是否有自定义逻辑注解,如果存在则执行自定义逻辑, 否则采用公共逻辑
        ModuleExecute moduleExecute = method.getDeclaredAnnotation(ModuleExecute.class);
        if (moduleExecute != null) {
            Class<? extends IModuleExecutor> executorClass = moduleExecute.execute();
            IModuleExecutor moduleExecutor = ReflectUtils.newInstance(executorClass);
            return moduleExecutor.execute(signType, domain, servicesContainer);
        } else {
            return execute(domain, httpFunctionMap.get(domain.getHttpMethod()));
        }
    }

    protected ServicesContainer servicesContainer;


    protected Map<HttpMethod, Function<HttpRequest, HttpResponse>> httpFunctionMap;

    public AbstractModuleImplementProxy(ServicesContainer servicesContainer, Enums.SIGN_TYPE signType) {
        this.servicesContainer = servicesContainer;
        this.httpFunctionMap = initHttpFunctionMap();
        this.signType = signType;
    }

    private Map<HttpMethod, Function<HttpRequest, HttpResponse>> initHttpFunctionMap() {
        Map<HttpMethod, Function<HttpRequest, HttpResponse>> map = new HashMap<>(4);
        map.put(HttpMethod.GET, servicesContainer.getHttpClientManager()::get);
        map.put(HttpMethod.POST, servicesContainer.getHttpClientManager()::post);
        map.put(HttpMethod.PUT, servicesContainer.getHttpClientManager()::put);
        map.put(HttpMethod.DELETE, servicesContainer.getHttpClientManager()::delete);
        return map;
    }

    protected ResponseBody execute(BaseRequest domain, Function<HttpRequest, HttpResponse> httpFunction) {
        SignatureBuilder signatureBuilder = this.signatureBuilder(domain);

        String signature = signatureBuilder.build();
        Events.dispatch(new DomainSignatureEvent(signature, signatureBuilder.getStringToSign()));

        HttpRequestBuilder httpRequestBuilder = this.requestBuilder(domain, signature);

        // wrapper http parameters
        HttpRequest httpRequest = httpRequestBuilder.build();
        // 发送Http请求
        HttpResponse httpResponse = httpFunction.apply(httpRequest);

        return parseResponseBody(httpResponse);
    }

    protected ResponseBody parseResponseBody(HttpResponse httpResponse) {
        // 解析接收到的响应结果
        try {
//            System.out.println(httpResponse.getBodyResource().asCharSource(StandardCharsets.UTF_8).read());
            ResponseBody responseBody = servicesContainer.getMessageConverter().getJsonSerializer().parse(httpResponse.getBodyResource().read(), ResponseBody.class);
            httpResponse.setBody(responseBody);
            responseBody.setHeaders(httpResponse.getHeaders());
            httpResponse.getHeaders().get("Request_Id").stream().findFirst().ifPresent(responseBody::setRequestId);
            httpResponse.getHeaders().get("X-Ca-Request-Id").stream().findFirst().ifPresent(responseBody::setTraceId);


            return responseBody;
        } catch (IOException | RuntimeException e) {
            try {
                String source = httpResponse.getBodyResource().asCharSource(StandardCharsets.UTF_8).read();
                ResponseBodyString responseBody = new ResponseBodyString(source);
                httpResponse.setBody(responseBody);
                StatusCode statusCode = new StatusCode();
                statusCode.setStatusCode(HttpStatus.SC_INTERNAL_SERVER_ERROR);
                statusCode.setReasonPhrase(e.getMessage());
                httpResponse.setHttpStatus(statusCode);

                return responseBody;
            } catch (IOException ex) {
                throw new RuntimeException(ex);
            }
        }
    }

    /**
     * 由子类创建实际的构造器
     * @return
     */
    protected abstract SignatureBuilder createSignatureBuilder(BaseRequest domain);

    protected abstract HttpRequestBuilder createRequestBuilder(BaseRequest domain);

    /**
     * 产生一个计算参数构造器
     * 并且自动设置请求Url及进行签名计算
     * @return
     */
    public SignatureBuilder signatureBuilder(BaseRequest domain) {
        SignatureBuilder signatureBuilder = createSignatureBuilder(domain);
        signatureBuilder.signature();

        return signatureBuilder;
    }

    public HttpRequestBuilder requestBuilder(BaseRequest domain, String signature) {
        HttpRequestBuilder httpRequestBuilder = createRequestBuilder(domain)
                .requestUrl()
                .headers(signature);

        if (HttpMethod.GET.equals(domain.getHttpMethod())) {
            httpRequestBuilder.queryParams();
        } else {
            httpRequestBuilder.body();
        }
        return httpRequestBuilder;
    }
}
