package net.linksfield.cube.partnersdk.sdk.modules.usage;

import com.google.common.collect.Multimap;
import lombok.Getter;
import net.linksfield.cube.partnersdk.configuration.EndpointPropertiesProxy;
import net.linksfield.cube.partnersdk.domain.BaseRequest;
import net.linksfield.cube.partnersdk.rest.HttpMethod;

import java.util.Map;

/**
 * @ClassName GetSimUsage
 * @Description Usage - get sim usage
 * @Author James.hu
 * @Date 2023/3/21
 **/
@Getter
public class GetSimUsage extends BaseRequest {
    /**
     * SIM ICCID
     */
    private String simId;

    /**
     * Category
     * <p>enum , supported: data </p>
     */
    private String category;

    /**
     * period type
     * <p>enum , supported: 1-daily，2-monthly</p>
     */
    private String periodType;

    /**
     * begin from
     * <p>When period_type=1，sample value: 2022-10-01. When period_type=2，sample value: 2022-10</p>
     */
    private String beginFrom;

    /**
     * end by
     * <p>When period_type=1，sample value: 2022-10-01. When period_type=2，sample value: 2022-10</p>
     */
    private String endBy;

    /**
     * Get SIM Usage
     * <p>Notes</p>
     * <ul>
     * <li>category :currently support: data.</li>
     * <li>period_type: daily/monthly</li>
     * <li>Daily: back track up to 60 days</li>
     * <li>Monthly: back track up to 6 months including current month</li>
     * <li>The CDRs may experience latency - the usage may change.</li>
     * <li>Only usage after 2023-01-01 is available.</li>
     * </ul><br/>
     * <p>Example</p>
     * <p>1：Query daily usage of 10/1/22.</p>
     * /sims/89852123456789012345/usage?category=data&period_type=1&begin_from=2022-10-01&end_by=2022-10-01
     * <p>2：Query monthly usage form 10/22 to 11/22.</p>
     * /sims/89852123456789012345/usage?category=data&period_type=2&begin_from=2022-10&end_by=2022-11
     * @param nonce
     * @param simId SIM ICCID
     */
    public GetSimUsage(int nonce, String simId, String category, String periodType, String beginFrom, String endBy) {
        super(HttpMethod.GET, nonce);
        this.simId = simId;
        this.category = category;
        this.periodType = periodType;
        this.beginFrom = beginFrom;
        this.endBy = endBy;
    }

    @Override
    public String requestUrl(EndpointPropertiesProxy endpointPropertiesProxy) {
        return endpointPropertiesProxy.getUsage().getSimUsage(simId);
    }

    @Override
    public void addUrlSignatureParameters(Map<String, Object> mapToSign) {
        mapToSign.put("sim_id", simId);
        mapToSign.put("category", category);
        mapToSign.put("period_type", periodType);
        mapToSign.put("begin_from", beginFrom);
        mapToSign.put("end_by", endBy);
    }

    @Override
    public void addQueryParams(Multimap<String, String> queryParams) {
        queryParams.put("category", category);
        queryParams.put("period_type", periodType);
        queryParams.put("begin_from", beginFrom);
        queryParams.put("end_by", endBy);
    }

    @Override
    public void addBody(Map<String, Object> body) {

    }
}
