package net.linksfield.cube.partnersdk.sdk.modules.sms;

import com.google.common.collect.Multimap;
import lombok.Data;
import lombok.Getter;
import lombok.Setter;
import net.linksfield.cube.partnersdk.configuration.EndpointPropertiesProxy;
import net.linksfield.cube.partnersdk.domain.BaseRequest;
import net.linksfield.cube.partnersdk.rest.HttpMethod;

import java.util.List;
import java.util.Map;

/**
 * @ClassName SendSms
 * @Description SMS - send sms
 * @Author James.hu
 * @Date 2023/3/17
 **/
@Getter
@Setter
public class SendSms extends BaseRequest {
    private List<String> simIds;

    private Message message;

    /**
     * <p><h1>Notes</h1>
     * <ul>
     * <li>The sending is asynchronous, and the request may fail</li>
     * <li>A maximum of 100 SIMs are supported</li>
     * <li>Each message contains a maximum of 140 characters</li>
     * <li>The retention time of the SMS in SMS-Center varies from 24 hours to 7 days</li>
     * <li>Not all SIMs can receive SMS via this API. For details, consult FAE</li>
     * </ul>
     * </p><br/>
     *
     * <p><h1>Attention</h1>
     * <ul>
     * <li>Sending may fail in below situations and the SMS will NOT be sent:</li>
     *
     * <ul>
     * <li>SMS service is not enabled for the organisation (consult FAE)</li>
     * <li>SIM IDs provided do not belong to the organisation</li>
     * <li>SIM ID duplicates</li>
     * <li>SMS sending via API is not supported for some SIMs</li>
     *  </ul>
     *
     * <li>Idempotence</li>
     *
     * <ul><li>If same nonce, same sim_ids and the same sms message is sent in 2 requests within 10s, they will be treated as the same request, and the same job id will be returned. The actual sending will only occur once.</li></ul>
     * <li>When a request is timeout, you may use above method to retrieve the job id.</li>
     *
     * <ul><li>Otherwise, it will be treated as a new request and the sms will be sent again.</li></ul>
     *
     * <li>Result</li>
     *
     * <ul>
     *     <li>Response of this API indicates the receipt of the request only. It does not indicate the status of sending of the SMS.</li>
     * <li>To request the status of the sending:</li>
     * <ul>
     * <li>List all SMS by Request ID (MT)</li>
     * <li>List all SMS by SIM ID(MT)</li>
     * </ul>
     * </ul>
     * </p>
     * @param nonce
     * @param simIds iccids
     * @param message type and content
     */
    public SendSms(int nonce, List<String> simIds, Message message) {
        super(HttpMethod.POST, nonce);
        this.simIds = simIds;
        this.message = message;
    }

    private SendSms(int nonce) {
        super(HttpMethod.POST, nonce);
    }

    public static SendSmsBuilder builder(int nonce) {
        SendSms domain = new SendSms(nonce);
        SendSmsBuilder builder = new SendSmsBuilder(domain);
        return builder;
    }

    @Override
    public String requestUrl(EndpointPropertiesProxy endpointPropertiesProxy) {
        return endpointPropertiesProxy.getSms().sendSms();
    }

    @Override
    public void addUrlSignatureParameters(Map<String, Object> mapToSign) {

    }

    @Override
    public void addQueryParams(Multimap<String, String> queryParams) {

    }

    @Override
    public void addBody(Map<String, Object> body) {
        body.put("sim_ids", simIds);
        body.put("message", message);
    }

    @Data
    public static class Message {
        private String content;
        private int type;
    }
}
