package net.linksfield.cube.partnersdk.sdk.modules.sms;

import com.google.common.collect.Multimap;
import lombok.Getter;
import lombok.Setter;
import net.linksfield.cube.partnersdk.configuration.EndpointPropertiesProxy;
import net.linksfield.cube.partnersdk.domain.BaseRequestV2;
import net.linksfield.cube.partnersdk.rest.HttpMethod;

import java.util.Map;

/**
 * @ClassName ListMTBySimId
 * @Description SMS - Retrieve a list of SMS(MT) sent to a single SIM
 * @Author James.hu
 * @Date 2023/3/22
 **/
@Getter
@Setter
public class ListMTBySimId extends BaseRequestV2 {
    /** SIM ICCID */
    private String simId;

    /** query status, enum, 1-success, 2-failed, 3-processing, 4-all . default 4 */
    private String status;

    /** Long, Epoch timestamp UTC 0 in millisecond; default value is 7 days prior to the current time */
    private String beginFrom;

    /** Long, Epoch timestamp UTC 0 in millisecond; default current time */
    private String endBy;

    /** Page No default is 1 */
    private Integer pageNo;

    /** Page Size default is 100, range 1-100 */
    private Integer pageSize;

    /**
     * Retrieve a list of SMS(MT) sent to a single SIM
     *
     * <h1>Notes：</h1>
     * <ul>
     * <li>SMS records are kept for 180 days at most</li>
     * <li>SIM ID provided must belong to the organisation of the request</li>
     * <li>By default ( if not present), the query time period is 7 days prior to the request.</li>
     * </ul>
     * @param nonce
     * @param simId SIM ICCID
     * @param status query status, enum, 1-success, 2-failed, 3-processing, 4-all . default 4
     * @param beginFrom Long, Epoch timestamp UTC 0 in millisecond; default value is 7 days prior to the current time
     * @param endBy Long, Epoch timestamp UTC 0 in millisecond; default current time
     * @param pageNo Page No default is 1
     * @param pageSize Page Size default is 100, range 1-100
     */
    public ListMTBySimId(int nonce, String simId, String status, String beginFrom, String endBy, Integer pageNo, Integer pageSize) {
        super(HttpMethod.GET, nonce);
        this.simId = simId;
        this.status = status;
        this.beginFrom = beginFrom;
        this.endBy = endBy;
        this.pageNo = pageNo;
        this.pageSize = pageSize;
    }

    private ListMTBySimId(int nonce) {
        super(HttpMethod.GET, nonce);
    }

    public static ListMTBySimIdBuilder builder(int nonce, String simId) {
        return new ListMTBySimIdBuilder(new ListMTBySimId(nonce), simId);
    }

    @Override
    public String requestUrl(EndpointPropertiesProxy endpointPropertiesProxy) {
        return endpointPropertiesProxy.getSms().listMTBySimId(simId);
    }

    @Override
    public void addQueryParams(Multimap<String, String> queryParams) {
        addOptionalQueryParam(queryParams, "status", status);
        addOptionalQueryParam(queryParams, "begin_from", beginFrom);
        addOptionalQueryParam(queryParams, "end_by", endBy);
        addOptionalQueryParam(queryParams, "page_no", pageNo);
        addOptionalQueryParam(queryParams, "page_size", pageSize);
    }

    @Override
    public void addBody(Map<String, Object> body) {

    }


}
