package net.linksfield.cube.partnersdk.sdk.modules.service;

import com.google.common.collect.Multimap;
import lombok.Getter;
import net.linksfield.cube.partnersdk.configuration.EndpointPropertiesProxy;
import net.linksfield.cube.partnersdk.domain.BaseRequest;
import net.linksfield.cube.partnersdk.rest.HttpMethod;

import java.util.Map;

/**
 * @ClassName ResetAsync
 * @Description Service - add bundle
 * @Author James.hu
 * @Date 2023/3/21
 **/
@Getter
public class AddBundle extends BaseRequest {
    /**
     * SIM ICCID
     */
    private String simId;

    /** Bundle ID */
    private String bundleId;

    /** Bundle type：10 - Basic、11 - Add on */
    private int bundleType;

    /** Number of add cycles. This value does not take effect when the type is 11- add on. If the type is 10 - basic, the value is mandatory. If it is empty, the value is 1 by default */
    private int cycles;

    /**
     * <p>Basic bundle rules
     * <ul>
     * <li>Add bundle will not change the current SIM status.</li>
     * <li>One SIM can have multiple basic bundles.</li>
     * <li>Only one basic bundle can be used at a time.</li>
     * <li>If multiple basic bundles are added to the SIM, the earlier bundle will be used by default (FIFO) ( when being activated).</li>
     * <li>When SIM is suspended, the service will be temporarily stopped and the service validity will still decrease with time.</li>
     * <li>When SIM status is Terminated, this API will not work.</li>
     * </ul>
     * </p>
     * <p>
     * Add-on Bundle - Pending
     * If the data of the SIM is insufficient in current cycle, add-on bundle will increase the data volume before the cycle ends. It dose not replace the basic bundle in use.
     * </p>
     * <p>
     *     Add on bundle rules
     * <li>SIM status must be Active.</li>
     * <li>The basic bundle must support add on bundle.</li>
     * <li>The add on bundle start immediately after being added.</li>
     * <li>When SIM is suspended, the add-on bundle will be removed.</li>
     * <li>When SIM is resumed, the add-on cannot be restored.</li>
     * </p>
     * @param nonce
     * @param simId SIM ICCID
     * @param bundleId Bundle ID
     * @param bundleType Bundle type：10 - Basic、11 - Add on
     * @param cycles Number of add cycles. This value does not take effect when the type is 11- add on. If the type is 10 - basic, the value is mandatory. If it is empty, the value is 1 by default
     */
    public AddBundle(int nonce, String simId, String bundleId, int bundleType, int cycles) {
        super(HttpMethod.POST, nonce);
        this.simId = simId;
        this.bundleId = bundleId;
        this.bundleType = bundleType;
        this.cycles = cycles;
    }

    @Override
    public String requestUrl(EndpointPropertiesProxy endpointPropertiesProxy) {
        return endpointPropertiesProxy.getService().addBundle(simId);
    }

    @Override
    public void addUrlSignatureParameters(Map<String, Object> mapToSign) {
        mapToSign.put("sim_id", simId);
        mapToSign.put("bundle_id", bundleId);
        mapToSign.put("bundle_type", bundleType);
        mapToSign.put("cycles", cycles);
    }

    @Override
    public void addQueryParams(Multimap<String, String> queryParams) {

    }

    @Override
    public void addBody(Map<String, Object> body) {
        body.put("sim_id", simId);
        body.put("bundle_id", bundleId);
        body.put("bundle_type", bundleType);
        body.put("cycles", cycles);
    }
}
