package net.linksfield.cube.partnersdk.sdk.modules.orders;

/**
 * @ClassName PlaceOrderBuilder
 * @Description Orderlite - place an order builder
 * @Author James.hu
 * @Date 2023/3/22
 **/
public class PlaceOrderBuilder {
    private PlaceOrder placeOrder;

    PlaceOrderBuilder(PlaceOrder placeOrder) {
        this.placeOrder = placeOrder;
    }

    /**
     * Add a product that needs to be ordered
     * @param sku Product code
     * @param quantity Quantity ordered
     * @param price Unit price
     * @return this
     */
    public PlaceOrderBuilder addItem(String id, int quantity, int price) {
        placeOrder.skus.add(new PlaceOrder.Sku(id, quantity, price));
        return this;
    }

    /**
     * Add accounting information
     * @param currency Currency code IOS | Enum: CNY
     * @param payChannel Enum: 1 - Postpaid billing
     * @return this
     */
    public PlaceOrderBuilder accounting(String currency, int payChannel) {
        placeOrder.accounting = new PlaceOrder.Accounting(currency, payChannel);
        return this;
    }

    /**
     * Add accounting information
     * @param currency Currency code IOS | Enum: CNY
     * @param payChannel Enum: 1 - Postpaid billing
     * @param couponId Coupon code
     * @return this
     */
    public PlaceOrderBuilder accounting(String currency, int payChannel, String couponId) {
        placeOrder.accounting = new PlaceOrder.Accounting(currency, payChannel, couponId);
        return this;
    }

    /**
     * Add accounting information
     * @param currency Currency code IOS | Enum: CNY
     * @param payChannel Enum: 1 - Postpaid billing
     * @param couponId Coupon code
     * @param discount Discount | Example: 6% = 6
     * @return this
     */
    public PlaceOrderBuilder accounting(String currency, int payChannel, String couponId, Integer discount) {
        placeOrder.accounting = new PlaceOrder.Accounting(currency, payChannel, couponId, discount);
        return this;
    }

    /**
     * Additional customer data
     * @param po Customer system serial number
     * @param remark Remark
     * @return this
     */
    public PlaceOrderBuilder extra(String po, String remark) {
        placeOrder.extra = new PlaceOrder.Extra(po, remark);
        return this;
    }

    /**
     * Add shipping address information
     * @param country Country
     * @param province Province
     * @param city City
     * @param district District
     * @param town Town
     * @param detail Full address
     * @param zip Zip code
     * @param specialInstructions Logistics extensions
     * @return this
     */
    public PlaceOrderBuilder address(String country, String province, String city, String district, String town, String detail, String zip, String specialInstructions) {
        placeOrder.address = new PlaceOrder.DeliveryAddress(country, province, city, district, town, detail, zip, specialInstructions);
        return this;
    }

    /**
     * Add shipping address information
     * @param country Country
     * @param province Province
     * @param city City
     * @param detail Full address
     * @return this
     */
    public PlaceOrderBuilder address(String country, String province, String city, String detail) {
        placeOrder.address = new PlaceOrder.DeliveryAddress(country, province, city, detail);
        return this;
    }

    /**
     * Add consignee information
     * @param name Name or nickname
     * @param mobileNo Mobile phone number
     * @return this
     */
    public PlaceOrderBuilder consignee(String name, String mobileNo) {
        placeOrder.contact = new PlaceOrder.Contact(name, mobileNo);
        return this;
    }

    /**
     * Add consignee information
     * @param name Name or nickname
     * @param telArea Telephone area code
     * @param telCode Landline phone number
     * @param mobileNo Mobile phone number
     * @param mobileArea Phone number country code
     * @param email Email
     * @return this
     */
    public PlaceOrderBuilder consignee(String name, String telArea, String telCode, String mobileNo, String mobileArea, String email) {
        placeOrder.contact = new PlaceOrder.Contact(name, telArea, telCode, mobileNo, mobileArea, email);
        return this;
    }

    public PlaceOrder build() {
        return this.placeOrder;
    }

}
