package net.linksfield.cube.partnersdk.sdk.modules.cdr;

import com.google.common.collect.Multimap;
import lombok.Getter;
import lombok.Setter;
import net.linksfield.cube.partnersdk.configuration.EndpointPropertiesProxy;
import net.linksfield.cube.partnersdk.domain.BaseRequest;
import net.linksfield.cube.partnersdk.rest.HttpMethod;

import java.util.Map;

/**
 * @ClassName SimCdr
 * @Description Cdr - Get SIM CDRs
 * @Author James.hu
 * @Date 2023/3/17
 **/
@Getter
@Setter
public class SimCdr extends BaseRequest {

    /** SIM ICCID */
    private String simId;

    /**
     * Start time
     */
    private String receivedStartTime;

    /**
     * End by
     */
    private String receivedEndBy;

    /**
     * Page No
     */
    private Integer pageNo;

    /**
     * Page Size
     */
    private Integer pageSize;

    /**
     * <p>Query CDR records</p>
     * <h1>Note</h1>
     * <ul>
     * <li>CDR can be back tracked for maximum 180 days.</li>
     * <li>This API can only query CDR since 2023-Jan-1.</li>
     * </ul>
     * @param nonce
     * @param simId SIM ICCID
     * @param receivedStartTime Start time
     * @param receivedEndBy End by
     * @param pageNo Page no
     * @param pageSize Page size
     */
    public SimCdr(int nonce, String simId, String receivedStartTime, String receivedEndBy, Integer pageNo, Integer pageSize) {
        super(HttpMethod.GET, nonce);
        this.simId = simId;
        this.receivedStartTime = receivedStartTime;
        this.receivedEndBy = receivedEndBy;
        this.pageNo = pageNo;
        this.pageSize = pageSize;
    }

    SimCdr(int nonce) {
        super(HttpMethod.GET, nonce);
    }

    public static SimCdrBuilder builder(int nonce, String simId) {
        return new SimCdrBuilder(new SimCdr(nonce), simId);
    }

    @Override
    public String requestUrl(EndpointPropertiesProxy endpointPropertiesProxy) {
        return endpointPropertiesProxy.getCdr().getBySimId(simId);
    }

    @Override
    public void addUrlSignatureParameters(Map<String, Object> mapToSign) {
        mapToSign.put("sim_id", simId);
        if (receivedStartTime != null) {
            mapToSign.put("received_start_time", receivedStartTime);
        }
        if (receivedEndBy != null) {
            mapToSign.put("received_end_by", receivedEndBy);
        }
        if (pageNo != null) {
            mapToSign.put("page_no", pageNo);
        }
        if (pageSize != null) {
            mapToSign.put("page_size", pageSize);
        }
    }

    @Override
    public void addQueryParams(Multimap<String, String> queryParams) {
        if (receivedStartTime != null) {
            queryParams.put("received_start_time", receivedStartTime);
        }
        if (receivedEndBy != null) {
            queryParams.put("received_end_by", receivedEndBy);
        }
        if (pageNo != null) {
            queryParams.put("page_no", pageNo.toString());
        }
        if (pageSize != null) {
            queryParams.put("page_size", pageSize.toString());
        }
    }

    @Override
    public void addBody(Map<String, Object> body) {

    }

}
