package net.linksfield.cube.partnersdk.sdk;

import net.linksfield.cube.partnersdk.domain.CommonVariables;
import net.linksfield.cube.partnersdk.event.Events;
import net.linksfield.cube.partnersdk.event.events.SdkInitEvent;
import net.linksfield.cube.partnersdk.sdk.proxy.ProxyHelper;

import java.io.Closeable;
import java.io.IOException;

/**
 * @ClassName PartnerClientImpl
 * @Description Sdk入口
 * @Author James.hu
 * @Date 2023/3/14
 **/
public final class PartnerClient implements Closeable {
    /** 公共配置 */
    private CommonVariables commonVariables;

    /** 业务处理逻辑包装 */
    private ServicesContainer servicesContainer;

    /**
     * Initialize the SDK
     * @param accessKey AccessKey
     * @param privateKey PrivateKey
     */
    public PartnerClient(String accessKey, String privateKey) {
        // 初始化公共参数
        this.commonVariables = new CommonVariables(accessKey, privateKey);
        init(commonVariables);
    }

    /**
     * Initialize the SDK
     * Customize the default request headers
     * @param accessKey AccessKey
     * @param privateKey PrivateKey
     * @param acceptLanguage Overwrite request header Accept-Language
     */
    public PartnerClient(String accessKey, String privateKey, String acceptLanguage) {
        // 初始化公共参数
        this.commonVariables = new CommonVariables(accessKey, privateKey);
        this.commonVariables.setAcceptLanguage(acceptLanguage);
        init(commonVariables);
    }

    private void init(CommonVariables commonVariables) {
        this.commonVariables = commonVariables;
        ServicesContainer servicesContainer = new ServicesContainer(this.commonVariables);
        this.servicesContainer = servicesContainer;

        // 初始化SDK
        initSdk(servicesContainer);

        Events.dispatch(new SdkInitEvent(this.commonVariables.getAccessKey(), this.commonVariables.getPrivateKey()));
    }

    private SdkV1 sdkV1;

    /**
     * Call the version 1.0 signature scheme
     * @return SDK
     */
    public SdkV1 v1() {
        return sdkV1;
    }

    private SdkV2 sdkV2;

    /**
     * Call the version 2.0 signature scheme
     * @return SDK
     */
    public SdkV2 v2() {
        return sdkV2;
    }

    private void initSdk(ServicesContainer servicesContainer) {
        this.sdkV1 = ProxyHelper.createSdkV1Proxy(servicesContainer);

        this.sdkV2 = ProxyHelper.createSdkV2Proxy(servicesContainer);
    }


    /**
     * Close and release Http request client (maybe httpclient5 httpclient5 or okhttp)
     */
    @Override
    public void close() throws IOException {
        servicesContainer.getHttpClientManager().close();
        Events.stop();
    }
}
