package net.linksfield.cube.partnersdk.rest.okhttp;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.io.ByteSource;
import lombok.extern.slf4j.Slf4j;
import net.linksfield.cube.partnersdk.rest.*;
import net.linksfield.cube.partnersdk.utils.GuavaUtils;
import okhttp3.MediaType;
import okhttp3.*;

import java.io.IOException;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * @ClassName OkHttpTemplate
 * @Description okHttp 请求接口
 * @Author James.hu
 * @Date 2023/3/22
 **/
@Slf4j
public class OkHttpTemplate implements HttpFunction {
    private OkHttpClient okHttpClient;

    public static final MediaType JSON = MediaType.parse("application/json; charset=utf-8");

    private OkHttpClient initOkHttpClient() {
        OkHttpClient okHttpClient = new OkHttpClient.Builder()
                .connectTimeout(HttpClientManager.HTTP_CONNECT_TIMEOUT, TimeUnit.SECONDS)
                .readTimeout(HttpClientManager.HTTP_READ_TIMEOUT, TimeUnit.SECONDS)
                .build();
        return okHttpClient;
    }

    public OkHttpTemplate() {
        log.debug("[Cube] The SDK will use OkHttp as the Http client implementation");
        this.okHttpClient = initOkHttpClient();
    }

    @Override
    public HttpResponse get(HttpRequest request) {
        Request.Builder requestBuilder = new Request.Builder().get();
        HttpUrl.Builder urlBuilder = HttpUrl.parse(request.getUrl()).newBuilder();
        request.getQueryParams().forEach((key, value) -> urlBuilder.addQueryParameter(key, value));

        Request okRequest = buildRequest(requestBuilder, urlBuilder.build(), request);
        return execute(okRequest);
    }

    @Override
    public HttpResponse post(HttpRequest request) {
        return postPutDelete(request, new Request.Builder()::post);
    }

    @Override
    public HttpResponse put(HttpRequest request) {
        return postPutDelete(request, new Request.Builder()::put);
    }

    @Override
    public HttpResponse delete(HttpRequest request) {
        return postPutDelete(request, new Request.Builder()::delete);
    }

    private HttpResponse postPutDelete(HttpRequest request, Function<RequestBody, Request.Builder> builderFunction) {
        String contentType = GuavaUtils.getMultimapFirst(request.getHeaders(), HttpHeaders.ContentType);

        Request.Builder requestBuilder;
        if (net.linksfield.cube.partnersdk.rest.MediaType.X_WWW_FORM_URLENCODED.equalsIgnoreCase(contentType)) {
            requestBuilder = new Request.Builder().get();
            HttpUrl.Builder urlBuilder = HttpUrl.parse(request.getUrl()).newBuilder();
            request.getQueryParams().forEach((key, value) -> urlBuilder.addQueryParameter(key, value));
        } else {
            RequestBody requestBody = RequestBody.create(request.getBodyString(), JSON);
            requestBuilder = builderFunction.apply(requestBody);
        }

        Request okRequest = buildRequest(requestBuilder, HttpUrl.parse(request.getUrl()), request);
        return execute(okRequest);
    }


    private Request buildRequest(Request.Builder requestBuilder, HttpUrl url, HttpRequest request) {
        request.getHeaders().forEach((key, value) -> requestBuilder.addHeader(key, value));
        Request okRequest = requestBuilder.url(url).build();
        return okRequest;
    }

    public HttpResponse execute(Request okRequest) {
        try {
            Response okResponse = okHttpClient.newCall(okRequest).execute();

            HttpResponse response = new HttpResponse();

            StatusCode statusCode = new StatusCode();
            statusCode.setStatusCode(okResponse.code());

            response.setHttpStatus(statusCode);

            Multimap<String, String> headers = HashMultimap.create();
            okResponse.headers().forEach(pair -> {
                headers.put(pair.getFirst(), pair.getSecond());
            });
            response.setHeaders(headers);

            response.setBodyResource(ByteSource.wrap(okResponse.body().bytes()));
            return response;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() throws IOException {
        log.debug("now connection manager is okhttp, close and shutdown connection pool");
    }
}
