package net.linksfield.cube.partnersdk.rest.httpclient5;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;
import lombok.extern.slf4j.Slf4j;
import net.linksfield.cube.partnersdk.rest.*;
import net.linksfield.cube.partnersdk.utils.GuavaUtils;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.classic.methods.HttpUriRequestBase;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.Header;
import org.apache.hc.core5.http.HttpException;
import org.apache.hc.core5.http.io.HttpClientResponseHandler;
import org.apache.hc.core5.http.io.entity.StringEntity;
import org.apache.hc.core5.net.URIBuilder;

import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;

/**
 * @ClassName HttpClient5Template
 * @Description HttpClient5 请求接口
 * @Author James.hu
 * @Date 2023/3/16
 **/
@Slf4j
public class HttpClient5Template implements HttpFunction {
    private HttpClient5Configuration configuration;

    public HttpClient5Template(HttpClient5Configuration configuration) {
        log.debug("[Cube] The SDK will use HttpClient5 as the Http client implementation");
        this.configuration = configuration;
    }

    @Override
    public HttpResponse get(HttpRequest request) {
        // 构造请求Url
        try {
            URIBuilder uriBuilder = new URIBuilder(request.getUrl());
            request.getQueryParams().forEach((key, value) -> uriBuilder.addParameter(key, value));
            // 构造Get
            HttpGet get = new HttpGet(uriBuilder.build());

            addHeaders(get, request);

            return execute(get);
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public HttpResponse post(HttpRequest request) {
        return postPutDelete(request);
    }

    @Override
    public HttpResponse put(HttpRequest request) {
        return postPutDelete(request);
    }

    @Override
    public HttpResponse delete(HttpRequest request) {
        return postPutDelete(request);
    }

    private HttpResponse postPutDelete(HttpRequest request) {
        // 构造请求Url
        try {
            URIBuilder uriBuilder = new URIBuilder(request.getUrl());

            String contentType = GuavaUtils.getMultimapFirst(request.getHeaders(), HttpHeaders.ContentType);
            if (MediaType.X_WWW_FORM_URLENCODED.equalsIgnoreCase(contentType)) {
                request.getQueryParams().forEach((key, value) -> uriBuilder.addParameter(key, value));
            }

            // 构造http请求
            HttpUriRequestBase post = new HttpUriRequestBase(request.getMethod().name(), uriBuilder.build());

            addHeaders(post, request);
            if (MediaType.APPLICATION_JSON.equalsIgnoreCase(contentType)) {
                // 注入请求body
                addBody(post, request);
            }

            return execute(post);
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private void addHeaders(HttpUriRequestBase httpRequest, HttpRequest requestWrapper) {
        requestWrapper.getHeaders().forEach((key, value) -> httpRequest.addHeader(key, value));
    }

    private void addBody(HttpUriRequestBase httpRequest, HttpRequest requestWrapper) {
        httpRequest.setEntity(new StringEntity(requestWrapper.getBodyString(), ContentType.APPLICATION_JSON));
    }

    private HttpResponse execute(HttpUriRequestBase httpRequest) {
        // 发送请求
        // HttpClient 不能用 try-resource包装, 会把pool manager一起关闭
        CloseableHttpClient httpClient = configuration.getClient();
        try {
            HttpResponse httpResponse = httpClient.execute(httpRequest, new ResponseHandler());
            return httpResponse;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private class ResponseHandler implements HttpClientResponseHandler<HttpResponse> {
        @Override
        public HttpResponse handleResponse(ClassicHttpResponse classicHttpResponse) throws HttpException, IOException {
            HttpResponse response = new HttpResponse();

            StatusCode statusCode = new StatusCode();
            statusCode.setStatusCode(classicHttpResponse.getCode());
            statusCode.setReasonPhrase(classicHttpResponse.getReasonPhrase());
            response.setHttpStatus(statusCode);

            Multimap<String, String> headers = HashMultimap.create();
            for (Header header : classicHttpResponse.getHeaders()) {
                headers.put(header.getName(), header.getValue());
            }
            response.setHeaders(headers);

            try (InputStream inputStream = classicHttpResponse.getEntity().getContent()){
                byte[] bodyBytes = ByteStreams.toByteArray(inputStream);
                response.setBodyResource(ByteSource.wrap(bodyBytes));
            } catch (IOException e) {
                throw new RuntimeException(e);
            } catch (UnsupportedOperationException e) {
                throw new RuntimeException(e);
            }

            return response;
        }
    }

    @Override
    public void close() throws IOException {
        log.debug("now connection manager is httpclient5, close and shutdown connection pool");
        this.configuration.close();
    }
}
