package net.linksfield.cube.partnersdk.rest.httpclient5;

import net.linksfield.cube.partnersdk.rest.HttpClientManager;
import org.apache.hc.client5.http.config.ConnectionConfig;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.client5.http.io.HttpClientConnectionManager;
import org.apache.hc.client5.http.socket.ConnectionSocketFactory;
import org.apache.hc.client5.http.socket.PlainConnectionSocketFactory;
import org.apache.hc.client5.http.ssl.SSLConnectionSocketFactory;
import org.apache.hc.core5.http.config.Registry;
import org.apache.hc.core5.http.config.RegistryBuilder;
import org.apache.hc.core5.http.io.SocketConfig;
import org.apache.hc.core5.io.CloseMode;
import org.apache.hc.core5.pool.PoolConcurrencyPolicy;
import org.apache.hc.core5.pool.PoolReusePolicy;
import org.apache.hc.core5.ssl.SSLContexts;
import org.apache.hc.core5.util.TimeValue;
import org.apache.hc.core5.util.Timeout;

import javax.net.ssl.SSLContext;
import java.io.Closeable;
import java.io.IOException;

/**
 * @ClassName HttpClient5Configuration
 * @Description Http5 连接配置
 * @Author James.hu
 * @Date 2023/3/16
 **/
public class HttpClient5Configuration implements Closeable {
    private HttpClientConnectionManager connectionManager;
    private RequestConfig defaultRequestConfig;

    public HttpClient5Configuration() {
        this.connectionManager = createPoolingConnectionFactory();
        this.defaultRequestConfig = createRequestConfig();
    }

    public CloseableHttpClient getClient() {
        final CloseableHttpClient httpclient = HttpClients.custom()
                .setConnectionManager(connectionManager)
                .setDefaultRequestConfig(defaultRequestConfig)
                .build();
        return httpclient;
    }

    public HttpClientConnectionManager createPoolingConnectionFactory() {
        // SSL context for secure connections can be created either based on
        // system or application specific properties.
        final SSLContext sslContext = SSLContexts.createSystemDefault();

        // Create a registry of custom connection socket factories for supported
        // protocol schemes.
        final Registry<ConnectionSocketFactory> socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.INSTANCE)
                .register("https", new SSLConnectionSocketFactory(sslContext))
                .build();

        PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager(
                socketFactoryRegistry, PoolConcurrencyPolicy.STRICT, PoolReusePolicy.LIFO, TimeValue.ofMinutes(5)
        );

        // Configure the connection manager to use socket configuration either
        // by default or for a specific host.
        connManager.setDefaultSocketConfig(SocketConfig.custom()
                .setTcpNoDelay(true)
                .build());
        // Validate connections after 10 sec of inactivity
        connManager.setDefaultConnectionConfig(ConnectionConfig.custom()
                .setConnectTimeout(Timeout.ofSeconds(HttpClientManager.HTTP_CONNECT_TIMEOUT))
                .setSocketTimeout(Timeout.ofSeconds(HttpClientManager.HTTP_READ_TIMEOUT))
                .setValidateAfterInactivity(TimeValue.ofSeconds(10))
                .setTimeToLive(TimeValue.ofHours(1))
                .build());

        // Configure total max or per route limits for persistent connections
        // that can be kept in the pool or leased by the connection manager.
        connManager.setMaxTotal(10);
        connManager.setDefaultMaxPerRoute(10);

        return connManager;
    }

    public RequestConfig createRequestConfig() {
        // Create global request configuration
        return RequestConfig.DEFAULT;
    }

    @Override
    public void close() throws IOException {
        this.connectionManager.close(CloseMode.GRACEFUL);
    }
}
