package net.linksfield.cube.partnersdk.rest.httpclient4;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;
import lombok.extern.slf4j.Slf4j;
import net.linksfield.cube.partnersdk.rest.*;
import net.linksfield.cube.partnersdk.utils.GuavaUtils;
import org.apache.http.Header;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.*;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.function.Function;

/**
 * @ClassName HttpClient4Template
 * @Description HttpClient4 请求接口
 * @Author James.hu
 * @Date 2023/3/16
 **/
@Slf4j
public class HttpClient4Template implements HttpFunction {
    private HttpClient4Configuration configuration;

    public HttpClient4Template(HttpClient4Configuration configuration) {
        log.debug("[Cube] The SDK will use HttpClient4 as the Http client implementation");
        this.configuration = configuration;
    }

    @Override
    public HttpResponse get(HttpRequest request) {
        return getDelete(request, HttpGet::new);
    }

    @Override
    public HttpResponse post(HttpRequest request) {
        return postPut(request, HttpPost::new);
    }

    @Override
    public HttpResponse put(HttpRequest request) {
        return postPut(request, HttpPut::new);
    }

    @Override
    public HttpResponse delete(HttpRequest request) {
        return getDelete(request, HttpDelete::new);
    }

    private HttpResponse getDelete(HttpRequest request, Function<URI, ? extends HttpRequestBase> httpMethod) {
        // 构造请求Url
        try {
            URIBuilder uriBuilder = new URIBuilder(request.getUrl());
            request.getQueryParams().forEach((key, value) -> uriBuilder.addParameter(key, value));
            // 构造Get
            HttpRequestBase get = httpMethod.apply(uriBuilder.build());

            addHeaders(get, request);

            return execute(get);
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private HttpResponse postPut(HttpRequest request, Function<URI, ? extends HttpEntityEnclosingRequestBase> httpMethod) {
        // 构造请求Url
        try {
            URIBuilder uriBuilder = new URIBuilder(request.getUrl());

            String contentType = GuavaUtils.getMultimapFirst(request.getHeaders(), HttpHeaders.ContentType);

            if (MediaType.X_WWW_FORM_URLENCODED.equalsIgnoreCase(contentType)) {
                request.getQueryParams().forEach((key, value) -> uriBuilder.addParameter(key, value));
            }

            // 构造http请求
            HttpEntityEnclosingRequestBase post = httpMethod.apply(uriBuilder.build());

            addHeaders(post, request);

            if (MediaType.APPLICATION_JSON.equalsIgnoreCase(contentType)) {
                // 注入请求body
                addBody(post, request);
            }

            return execute(post);
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private void addHeaders(HttpRequestBase httpRequest, HttpRequest requestWrapper) {
        requestWrapper.getHeaders().forEach((key, value) -> httpRequest.addHeader(key, value));
    }

    private void addBody(HttpEntityEnclosingRequestBase httpRequest, HttpRequest requestWrapper) {
        StringEntity stringEntity = new StringEntity(requestWrapper.getBodyString(), ContentType.APPLICATION_JSON);
        httpRequest.setEntity(stringEntity);
    }

    private HttpResponse execute(HttpRequestBase httpRequest) {
        // 发送请求
        CloseableHttpClient httpClient = configuration.getClient();
        try {
            HttpResponse httpResponse = httpClient.execute(httpRequest, new ResponseHandler());
            return httpResponse;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() throws IOException {
        log.debug("now connection manager is httpclient4, close and shutdown connection pool");
        this.configuration.close();
    }

    private class ResponseHandler implements org.apache.http.client.ResponseHandler<HttpResponse> {
        @Override
        public HttpResponse handleResponse(org.apache.http.HttpResponse httpResponse) throws ClientProtocolException, IOException {
            HttpResponse response = new HttpResponse();

            StatusCode statusCode = new StatusCode();
            statusCode.setStatusCode(httpResponse.getStatusLine().getStatusCode());
            statusCode.setReasonPhrase(httpResponse.getStatusLine().getReasonPhrase());
            response.setHttpStatus(statusCode);

            Multimap<String, String> headers = HashMultimap.create();
            for (Header header : httpResponse.getAllHeaders()) {
                headers.put(header.getName(), header.getValue());
            }
            response.setHeaders(headers);

            try (InputStream inputStream = httpResponse.getEntity().getContent()){
                byte[] bodyBytes = ByteStreams.toByteArray(inputStream);
                response.setBodyResource(ByteSource.wrap(bodyBytes));
            } catch (IOException e) {
                throw new RuntimeException(e);
            } catch (UnsupportedOperationException e) {
                throw new RuntimeException(e);
            }

            return response;
        }
    }

}
