package net.linksfield.cube.partnersdk.rest;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import net.linksfield.cube.partnersdk.configuration.EndpointPropertiesProxy;
import net.linksfield.cube.partnersdk.domain.BaseRequest;
import net.linksfield.cube.partnersdk.sdk.ServicesContainer;

import java.util.HashMap;
import java.util.Map;

/**
 * @ClassName HttpRequestBuilder
 * @Description Http请求数据构造
 * @Author James.hu
 * @Date 2023/3/20
 **/
@Slf4j
public abstract class HttpRequestBuilder {
    protected BaseRequest domain;

    @Getter
    protected Map<String, Object> bodyMap;

    protected ServicesContainer servicesContainer;

    @Getter
    protected HttpRequest requestWrapper;

    public HttpRequestBuilder(BaseRequest domain, ServicesContainer servicesContainer) {
        this.domain = domain;
        this.servicesContainer = servicesContainer;
        this.requestWrapper = new HttpRequest();
        this.method(domain.getHttpMethod());
    }

    /** 设置请求方法 */
    public HttpRequestBuilder method(HttpMethod method) {
        this.requestWrapper.setMethod(method);
        return this;
    }

    /** 设置请求Url 回调给Domain来实现指定 */
    public HttpRequestBuilder requestUrl() {
        EndpointPropertiesProxy propertiesProxy = servicesContainer.getEndpointManager().getPropertiesProxy();
        String requestUrl = domain.requestUrl(propertiesProxy);
        if (log.isTraceEnabled()) {
            log.trace("domain:[{}] set request url to : {}", domain.getClass().getName(), requestUrl);
        }
        this.requestWrapper.setUrl(requestUrl);
        return this;
    }

    /** 配置Http Header */
    public HttpRequestBuilder headers(String signature) {
        Multimap<String, String> headers = HashMultimap.create();
        if (domain.getExtendHeaders() != null) {
            headers.putAll(domain.getExtendHeaders());
        }
        // 添加认证header
        headers.put("Authorization", "lf " + servicesContainer.getCommonVariables().getAccessKey() + "/" + signature);
        globalHeaders(headers);
        this.requestWrapper.setHeaders(headers);
        return this;
    }


    /** 由子类实现 构造公共请求头 */
    protected abstract void globalHeaders(Multimap<String, String> headers);

    /** 配置查询参数 */
    public HttpRequestBuilder queryParams() {
        Multimap<String, String> queryParams = HashMultimap.create();
        globalQueryParams(queryParams);
        domain.addQueryParams(queryParams);

        this.requestWrapper.setQueryParams(queryParams);
        return this;
    }

    public HttpRequestBuilder queryParam(String key, String value) {
        Multimap<String, String> queryParams = this.requestWrapper.getQueryParams();
        if (queryParams == null) {
            queryParams = HashMultimap.create();
        }
        queryParams.put(key, value);
        return this;
    }

    /** 由子类实现 构造公共请求参数 */
    protected abstract void globalQueryParams(Multimap<String, String> queryParams);

    public HttpRequestBuilder body() {
        Map<String, Object> body = new HashMap<>();
        globalBodyMap(body);
        domain.addBody(body);

        this.bodyMap = body;

        String bodyString = servicesContainer.getMessageConverter().getJsonSerializer().toJson(body);
        this.requestWrapper.setBodyString(bodyString);
        return this;
    }

    protected abstract void globalBodyMap(Map<String, Object> body);

    public HttpRequest build() {
        return requestWrapper;
    }
}
