package net.linksfield.cube.partnersdk.rest;

import lombok.Getter;
import net.linksfield.cube.partnersdk.event.Events;
import net.linksfield.cube.partnersdk.event.events.HttpRequestEvent;
import net.linksfield.cube.partnersdk.event.events.HttpResponseEvent;
import net.linksfield.cube.partnersdk.rest.httpclient4.HttpClient4Configuration;
import net.linksfield.cube.partnersdk.rest.httpclient4.HttpClient4Template;
import net.linksfield.cube.partnersdk.rest.httpclient5.HttpClient5Configuration;
import net.linksfield.cube.partnersdk.rest.httpclient5.HttpClient5Template;
import net.linksfield.cube.partnersdk.rest.okhttp.OkHttpTemplate;
import net.linksfield.cube.partnersdk.utils.ClassUtils;

import java.io.Closeable;
import java.io.IOException;
import java.util.function.Function;

/**
 * @ClassName HttpClientManager
 * @Description Http客户端管理器 由此管理不同的实现
 * @Author James.hu
 * @Date 2023/3/16
 **/
public class HttpClientManager implements HttpFunction {
    public static final int HTTP_CONNECT_TIMEOUT = 30;
    public static final int HTTP_READ_TIMEOUT = 30;

    private static final boolean httpClient5Present;

    private static final boolean httpClient4Present;

    private static final boolean okHttpPresent;

    @Getter
    private HttpFunction httpTemplate;

    static {
        // 检测支持的http客户端实现
        ClassLoader classLoader = HttpClientManager.class.getClassLoader();
        httpClient5Present = ClassUtils.isPresent("org.apache.hc.client5.http.impl.classic.CloseableHttpClient", classLoader) &&
                ClassUtils.isPresent("org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager", classLoader);

        httpClient4Present = ClassUtils.isPresent("org.apache.http.impl.client.CloseableHttpClient", classLoader) &&
                ClassUtils.isPresent("org.apache.http.impl.conn.PoolingHttpClientConnectionManager", classLoader);

        okHttpPresent = ClassUtils.isPresent("okhttp3.OkHttpClient", classLoader);
    }

    public HttpClientManager() {
        // 根据被检测结果, 决定采用的http客户端实现
        if (httpClient5Present) {
            httpTemplate = new HttpClient5Template(new HttpClient5Configuration());
        } else if (httpClient4Present) {
            httpTemplate = new HttpClient4Template(new HttpClient4Configuration());
        } else if(okHttpPresent) {
            httpTemplate = new OkHttpTemplate();
        } else {
            throw new IllegalArgumentException("Unable to determine valid http client library");
        }
    }

    public HttpResponse get(HttpRequest httpRequest) {
        return execute(getHttpTemplate()::get, httpRequest);
    }

    @Override
    public HttpResponse post(HttpRequest httpRequest) {
        return execute(getHttpTemplate()::post, httpRequest);
    }

    @Override
    public HttpResponse put(HttpRequest httpRequest) {
        return execute(getHttpTemplate()::put, httpRequest);
    }

    @Override
    public HttpResponse delete(HttpRequest httpRequest) {
        return execute(getHttpTemplate()::delete, httpRequest);
    }

    private HttpResponse execute(Function<HttpRequest, HttpResponse> httpFunction, HttpRequest request) {
        Events.dispatch(new HttpRequestEvent(request));

        HttpResponse response = httpFunction.apply(request);

        Events.dispatch(new HttpResponseEvent(response));

        return response;
    }

    @Override
    public void close() throws IOException {
        this.httpTemplate.close();
    }
}
