package net.linksfield.cube.partnersdk.event;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import lombok.extern.slf4j.Slf4j;

import java.util.Collection;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Dispatches {@link Event}s in a separate thread. Currently only single thread
 * does that. Potentially there could be multiple channels for each event type
 * class and a thread pool can be used to dispatch the events.
 * 异步调度器：
 * <ui>
 * <li>每一个事件类型允许注册一个或多个事件处理器</li>
 * <li>事件循环运行在单独的线程中</li>
 * <li>事件处理器与事件循环运行在不同的线程中（异步），同一个事件的多个事件处理器运行在同一线程中（同步）</li>
 * </ui>
 * @author
 */
@Slf4j
public class AsyncDispatcher implements Dispatcher {
    /**
     * 标记 表示事件处理器是否正常接收通知分发
     */
    private boolean running = false;

    /** 事件处理接口 任务池 */
    private final ExecutorService eventHandlingPool;
    /** 事件映射表 */
    protected final Multimap<EventType, EventHandler> eventDispatchers;

    /**
     * 注册一个事件发布器
     */
    public AsyncDispatcher() {
        this.eventDispatchers = HashMultimap.create();
        this.eventHandlingPool = Executors.newSingleThreadExecutor();
    }

    /**
     * 事件循环逻辑
     */
    private class EventProcessTask implements Runnable {
        private Event event;
        private Collection<EventHandler> handlers;

        public EventProcessTask(Event event, Collection<EventHandler> handlers) {
            this.event = event;
            this.handlers = handlers;
        }

        @Override
        public void run() {
            log.debug("Event process for [{}] by {} handlers", event.getType(), handlers.size());
            for (EventHandler eventHandler : handlers) {
                eventHandler.handle(event);
            }
        }
    }

    /**
     * 启动事件监听
     */
    public void serviceStart() {
        this.running = true;
    }

    /**
     * 关闭事件监听
     */
    public void serviceStop() {
        this.running = false;
        this.eventHandlingPool.shutdownNow();
    }

    /**
     * 分发函数，事件循环调用
     */
    public void dispatch(Event event) {
        if (log.isDebugEnabled()) {
            log.debug("Dispatching the event {}.{}", event.getClass().getName(), event.toString());
        }

        EventType eventType = event.getType();
        try {
            // 查找是否存在对应事件类型的实现
            if (eventDispatchers.containsKey(eventType) && running) {
                // 将此事件的所有处理进行分发
                Collection<EventHandler> handlers = eventDispatchers.get(eventType);

                // 提交事件处理到 worker 线程池
                eventHandlingPool.execute(new EventProcessTask(event, handlers));
            }
        } catch (Throwable t) {
            log.error("Event dispatcher throw error in {} , error:{}", getClass().getName(), t.getMessage());
        }
    }

    private void internalRegister(EventType eventType, EventHandler handler){
        log.info("Event Registering [{}] for {}", eventType, handler.getClass());
        eventDispatchers.put(eventType, handler);
    }

    /**
     * 注册事件处理器
     * @param eventType 事件类型
     * @param handler 事件处理器
     */
    @Override
    public void register(EventType eventType, EventHandler handler) {
        this.internalRegister(eventType, handler);
    }

}