
package net.lightapi.portal.user.query.handler;

import com.networknt.httpstring.AttachmentConstants;
import com.networknt.monad.Result;
import com.networknt.utility.NioUtils;
import com.networknt.rpc.HybridHandler;
import com.networknt.rpc.router.ServiceHandler;
import java.nio.ByteBuffer;
import java.util.Map;

import io.undertow.server.HttpServerExchange;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static net.lightapi.portal.user.query.UserQueryStreams.dbProvider;

/**
 * This endpoint will be called from the user-command to check if the userType and entityId exists in the system
 * when creating a new user. It queries the user database by calling the db provider. This endpoint is protected
 * as we don't want to reveal user's email to others.
 *
 * @author Steve Hu
 */
@ServiceHandler(id="lightapi.net/user/queryUserByTypeEntityId/0.1.0")
public class QueryUserByTypeEntityId implements HybridHandler {
    private static final Logger logger = LoggerFactory.getLogger(QueryUserByTypeEntityId.class);

    static final String USER_NOT_FOUND_BY_ID = "ERR11609";
    static final String USER_QUERY_DENIED = "ERR11618";

    @Override
    public ByteBuffer handle(HttpServerExchange exchange, Object input)  {
        if(logger.isTraceEnabled()) logger.trace("input = {}", input);
        Map<String, Object> map = (Map<String, Object>)input;
        String userType = (String)map.get("userType");
        String entityId = (String)map.get("entityId");
        if(logger.isTraceEnabled()) logger.trace("userType = {} entityId = {}", userType, entityId);
        String email = null;
        boolean isAdmin = false;
        // TODO need to debug to confirm the logic.
        Map<String, Object> auditInfo = exchange.getAttachment(AttachmentConstants.AUDIT_INFO);
        if(auditInfo != null) {
            email = (String)auditInfo.get("user_id");
            if(email != null) {
                // authorization code token make sure the userId is match or roles contains admin
                String roles = (String)auditInfo.get("roles");
                if(roles != null && roles.contains("admin")) {
                    isAdmin = true;
                }
            }
            // userId is null indicates it is a client credentials token from another service.
        }
        // email shouldn't be null as the schema validation is done.
        Result<String> result = dbProvider.queryUserByTypeEntityId(userType, entityId);
        if(result != null && result.isSuccess()) {
            if(email != null && !email.equals(result.getResult()) && !isAdmin) {
                // email and jwt userId is not matched and roles doesn't contain admin
                return NioUtils.toByteBuffer(getStatus(exchange, USER_QUERY_DENIED, email, entityId));
            }
            return NioUtils.toByteBuffer(result.getResult());
        } else {
            return NioUtils.toByteBuffer(getStatus(exchange, USER_NOT_FOUND_BY_ID, entityId));
        }
    }
}
