package net.lightapi.portal.user.query;

import com.networknt.config.Config;
import com.networknt.config.JsonMapper;
import com.networknt.kafka.common.KafkaStreamsConfig;
import com.networknt.kafka.streams.LightStreams;
import net.lightapi.portal.PortalConfig;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.common.serialization.Serdes;
import org.apache.kafka.streams.*;
import org.apache.kafka.streams.errors.StreamsUncaughtExceptionHandler;
import org.apache.kafka.streams.processor.AbstractProcessor;
import org.apache.kafka.streams.processor.Processor;
import org.apache.kafka.streams.processor.ProcessorContext;
import org.apache.kafka.streams.processor.ProcessorSupplier;
import org.apache.kafka.streams.state.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.LinkedList;
import java.util.Map;
import java.util.Properties;

public class NotificationStreams implements LightStreams {
    static private final Logger logger = LoggerFactory.getLogger(NotificationStreams.class);
    static final KafkaStreamsConfig streamsConfig = (KafkaStreamsConfig) Config.getInstance().getJsonObjectConfig(KafkaStreamsConfig.CONFIG_NAME, KafkaStreamsConfig.class);
    static final PortalConfig portalConfig = (PortalConfig) Config.getInstance().getJsonObjectConfig(PortalConfig.CONFIG_NAME, PortalConfig.class);

    static private final String notification = "user-notification-store";

    KafkaStreams notificationStreams;

    public NotificationStreams() {
        logger.info("NotificationStreams is created");
    }

    public ReadOnlyKeyValueStore<String, String> getNotificationStore() {
        QueryableStoreType<ReadOnlyKeyValueStore<String, String>> queryableStoreType = QueryableStoreTypes.keyValueStore();
        StoreQueryParameters<ReadOnlyKeyValueStore<String, String>> sqp = StoreQueryParameters.fromNameAndType(notification, queryableStoreType);
        return notificationStreams.store(sqp);
    }

    public KeyQueryMetadata getNotificationStreamsMetadata(String email) {
        return notificationStreams.queryMetadataForKey(notification, email, Serdes.String().serializer());
    }

    private void startNotificationStreams(String ip, int port) {

        final StreamsBuilder builder = new StreamsBuilder();

        StoreBuilder<KeyValueStore<String, String>> keyValueNotificationStoreBuilder =
                Stores.keyValueStoreBuilder(Stores.persistentKeyValueStore(notification),
                        Serdes.String(),
                        Serdes.String());
        builder.addStateStore(keyValueNotificationStoreBuilder);

        builder.stream(portalConfig.getNotificationTopic()).process(new ProcessorSupplier() {
            @Override
            public Processor get() {
                return new NotificationStreams.NotificationEventProcessor();
            }
        }, notification);

        final Topology topology = builder.build();

        Properties streamsProps = new Properties();
        streamsProps.putAll(streamsConfig.getProperties());
        streamsProps.put(StreamsConfig.DEFAULT_KEY_SERDE_CLASS_CONFIG, Serdes.ByteArray().getClass());
        streamsProps.put(StreamsConfig.DEFAULT_VALUE_SERDE_CLASS_CONFIG, Serdes.ByteArray().getClass());
        streamsProps.put(StreamsConfig.APPLICATION_ID_CONFIG, portalConfig.getNotificationApplicationId());
        streamsProps.put(StreamsConfig.APPLICATION_SERVER_CONFIG, ip + ":" + port);
        notificationStreams = new KafkaStreams(topology, streamsProps);
        notificationStreams.setUncaughtExceptionHandler(ex -> {
            logger.error("Kafka-Streams uncaught exception occurred. Stream will be replaced with new thread", ex);
            return StreamsUncaughtExceptionHandler.StreamThreadExceptionResponse.REPLACE_THREAD;
        });
       if(streamsConfig.isCleanUp()) {
            notificationStreams.cleanUp();
        }
        notificationStreams.start();
    }

    public static class NotificationEventProcessor extends AbstractProcessor<byte[], byte[]> {

        private ProcessorContext pc;
        private KeyValueStore<String, String> notificationStore;

        public NotificationEventProcessor() {
        }

        @Override
        public void init(ProcessorContext pc) {

            this.pc = pc;
            this.notificationStore = (KeyValueStore<String, String>) pc.getStateStore(notification);
            if(logger.isInfoEnabled()) logger.info("Processor initialized");
        }

        @Override
        public void process(byte[] key, byte[] value) {
            // process the notification to generate a list as value.
            String email = new String(key, StandardCharsets.UTF_8);
            String notification = new String(value, StandardCharsets.UTF_8);
            Map<String, Object> notificationMap = JsonMapper.string2Map(notification);
            LinkedList<Map<String, Object>> notificationList = new LinkedList<>();
            String notifications = notificationStore.get(email);
            if(notifications != null) {
                notificationList.addAll(JsonMapper.string2List(notifications));
            }
            notificationList.addFirst(notificationMap);
            if(notificationList.size() > 50) {
                notificationList.removeLast(); // we only keep the last 50 notifications.
            }
            notificationStore.put(email, JsonMapper.toJson(notificationList));
        }

        @Override
        public void close() {
            if(logger.isInfoEnabled()) logger.info("Closing processor...");
        }
    }
    @Override
    public void start(String ip, int port) {
        if(logger.isDebugEnabled()) logger.debug("NotificationStreams is starting...");
        startNotificationStreams(ip, port);
    }

    @Override
    public void close() {
        if(logger.isDebugEnabled()) logger.debug("NotificationStreams is closing...");
        notificationStreams.close();
    }
}
