
package net.lightapi.portal.user.query.handler;

import com.networknt.httpstring.AttachmentConstants;
import com.networknt.monad.Result;
import com.networknt.utility.NioUtils;
import com.networknt.rpc.HybridHandler;
import com.networknt.rpc.router.ServiceHandler;
import java.nio.ByteBuffer;
import java.util.Map;

import io.undertow.server.HttpServerExchange;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static net.lightapi.portal.user.query.UserQueryStreams.dbProvider;

/**
 * This is the endpoint used to get the email address from the Taiji wallet address. There are three scenarios:
 * 1. The same user to get the email address from the taiji address. This is a rare situation.
 * 2. Admin user use his/her credentials (authorization code token) to get another user's email from taiji address
 * 3. Access from other portal services with client credentials token.
 *
 * For security reason, one user cannot get other user's email address with the taiji address unless he/she is an admin
 *
 * @author Steve Hu
 *
 */
@ServiceHandler(id="lightapi.net/user/queryUserByWallet/0.1.0")
public class QueryUserByWallet implements HybridHandler {
    private static final Logger logger = LoggerFactory.getLogger(QueryUserByWallet.class);

    static final String EMAIL_NOT_FOUND_BY_WALLET = "ERR11630";
    static final String WALLET_QUERY_DENIED = "ERR11629";

    @Override
    public ByteBuffer handle(HttpServerExchange exchange, Object input)  {
        if(logger.isTraceEnabled()) logger.trace("input = " + input);
        Map<String, Object> map = (Map<String, Object>)input;
        String wallet = (String)map.get("taijiWallet");
        String email = null;
        boolean isAdmin = false;
        Map<String, Object> auditInfo = exchange.getAttachment(AttachmentConstants.AUDIT_INFO);
        if(auditInfo != null) {
            email = (String)auditInfo.get("user_id");
            if(email != null) {
                // authorization code token make sure the userId is match or roles contains admin
                String roles = (String)auditInfo.get("roles");
                if(roles != null && roles.contains("admin")) {
                    isAdmin = true;
                }
            }
            // userId is null indicates it is a client credentials token from another service.
        }
        // email shouldn't be null as the schema validation is done.
        Result<String> result = dbProvider.queryUserById(wallet);

        if(result != null && result.isSuccess()) {
            if(email != null && !email.equals(result.getResult()) && !isAdmin) {
                // email and jwt userId is not matched and roles doesn't contain admin
                return NioUtils.toByteBuffer(getStatus(exchange, WALLET_QUERY_DENIED, wallet, email));
            }
            return NioUtils.toByteBuffer(result.getResult());
        }
        return NioUtils.toByteBuffer(getStatus(exchange, EMAIL_NOT_FOUND_BY_WALLET, wallet));
    }
}
