
package net.lightapi.portal.user.query.handler;

import com.networknt.httpstring.AttachmentConstants;
import com.networknt.monad.Result;
import com.networknt.utility.NioUtils;
import com.networknt.rpc.HybridHandler;
import com.networknt.rpc.router.ServiceHandler;
import java.nio.ByteBuffer;
import java.util.Map;

import io.undertow.server.HttpServerExchange;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static net.lightapi.portal.user.query.UserQueryStreams.dbProvider;

/**
 * This endpoint will be called from the UI with a logged-in user to get his/her profile. It is also
 * being called from the user-command to check if a user exists before creating a new user. When called
 * from the UI, an authorization code token is required. When calling from another service, for example,
 * user-command service, a client credentials token is required. There would be no user_id and roles in
 * the client credentials token.
 *
 * As we need the latest info about the user to get retrieved, we need to query the database directly.
 *
 * @author Steve Hu
 */
@ServiceHandler(id="lightapi.net/user/queryUserByEmail/0.1.0")
public class QueryUserByEmail implements HybridHandler {
    private static final Logger logger = LoggerFactory.getLogger(QueryUserByEmail.class);

    static final String USER_NOT_FOUND_BY_EMAIL = "ERR11607";
    static final String USER_QUERY_DENIED = "ERR11618";
    @Override
    public ByteBuffer handle(HttpServerExchange exchange, Object input)  {
        if(logger.isTraceEnabled()) logger.trace("input = {}", input);
        Map<String, String> map = (Map<String, String>)input;
        String email = map.get("email");
        if(logger.isTraceEnabled()) logger.trace("email = {}", email);
        Map<String, Object> auditInfo = exchange.getAttachment(AttachmentConstants.AUDIT_INFO);
        // if security is not enabled, then allow everyone to access for dev env.
        if(auditInfo != null) {
            String userId = (String)auditInfo.get("user_id");
            if(userId != null) {
                // authorization code token make sure the userId is match or roles contains admin
                String roles = (String)auditInfo.get("roles");
                if(!email.equals(userId) && roles != null && !roles.contains("admin")) {
                    // email and jwt userId is not matched and roles doesn't contain admin
                    return NioUtils.toByteBuffer(getStatus(exchange, USER_QUERY_DENIED, email, userId));
                }
            }
            // userId is null indicates it is a client credentials token from another service.
        }
        // email shouldn't be null as the schema validation is done.
        // data doesn't have hashed password as it is never going to be returned in any API. only reset is allowed for maximum security.
        // don't ask us for your password as we cannot brute force it with our computer power :)
        Result<String> result = dbProvider.queryUserByEmail(email);
        if(result != null && result.isSuccess()) {
            return NioUtils.toByteBuffer(result.getResult());
        } else {
            // not found.
            return NioUtils.toByteBuffer(getStatus(exchange, USER_NOT_FOUND_BY_EMAIL, email));
        }
    }
}
