/*
 * Copyright (c) 2020 Network New Technologies Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.lightapi.portal.user.query.handler;

import com.networknt.config.JsonMapper;
import com.networknt.monad.Result;
import com.networknt.utility.HashUtil;
import com.networknt.utility.NioUtils;
import com.networknt.rpc.HybridHandler;
import com.networknt.rpc.router.ServiceHandler;
import java.nio.ByteBuffer;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.HashMap;
import java.util.Map;

import io.undertow.server.HttpServerExchange;

import static net.lightapi.portal.user.query.UserQueryStreams.dbProvider;

/**
 * This endpoint is called from a customized authenticator from light-oauth2. It returns two values, user
 * id and roles in string after successfully authenticated from the user_t table.
 *
 * The validatePassword method in HashUtil will be called to verify the password again the stored password
 *
 * When a new user is registered, the verification flag is false. It will be updated to true only after the
 * email address is confirmed. We check the flag during the login so a user must have a valid email address.
 *
 * @author Steve Hu
 */
@ServiceHandler(id="lightapi.net/user/loginUser/0.1.0")
public class LoginUser implements HybridHandler {
    private static final String ERROR_VALIDATE_PASSWORD = "ERR11616";
    private static final String USER_NOT_FOUND_BY_EMAIL = "ERR11607";
    private static final String WRONG_LOGIN_PASSWORD = "ERR11612";
    private static final String EMAIL_NOT_CONFIRMED = "ERR11610";

    @Override
    public ByteBuffer handle(HttpServerExchange exchange, Object input)  {
        Map<String, String> map = (Map<String, String>)input;
        String email = map.get("email");
        String password = map.get("password");
        // get user from user_t table
        Result<String> result = dbProvider.loginUserByEmail(email);
        if(result.isFailure()) {
            return NioUtils.toByteBuffer(getStatus(exchange, USER_NOT_FOUND_BY_EMAIL, email));
        } else {
            if(logger.isTraceEnabled()) logger.trace("user = {}", result.getResult());
            return NioUtils.toByteBuffer(validatePassword(exchange, password, result.getResult(), email));
        }
    }

    private String validatePassword(HttpServerExchange exchange, String inputPassword, String userResult, String email) {
        try {
            Map<String, Object> userMap = JsonMapper.string2Map(userResult);
            boolean valid = HashUtil.validatePassword(inputPassword.toCharArray(), (String)userMap.get("password"));
            if(valid) {
                // get the user roles from user_t table.
                boolean verified = (Boolean)userMap.get("verified");
                if(verified) {
                    return userResult;
                } else {
                    return getStatus(exchange, EMAIL_NOT_CONFIRMED, email);
                }
            } else {
                return getStatus(exchange, WRONG_LOGIN_PASSWORD, email);
            }
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            return getStatus(exchange, ERROR_VALIDATE_PASSWORD, email);
        }
    }
}
