package net.lightapi.portal.user.query;

import com.fasterxml.jackson.core.type.TypeReference;
import com.networknt.config.Config;
import com.networknt.config.JsonMapper;
import com.networknt.utility.UuidUtil;
import io.cloudevents.core.v1.CloudEventV1;
import net.lightapi.portal.PortalConstants;
import com.networknt.db.provider.DbProvider;
import com.networknt.kafka.common.KafkaStreamsConfig;
import com.networknt.kafka.streams.LightStreams;
import com.networknt.oas.model.Operation;
import com.networknt.oas.model.Path;
import com.networknt.oas.model.SecurityParameter;
import com.networknt.oas.model.SecurityRequirement;
import com.networknt.openapi.OpenApiHelper;
import com.networknt.service.SingletonServiceFactory;
import net.lightapi.portal.PortalConfig;
import net.lightapi.portal.PortalUtil;
import net.lightapi.portal.db.PortalDbProvider;
import org.apache.kafka.common.serialization.Serdes;
import org.apache.kafka.streams.*;
import org.apache.kafka.streams.errors.StreamsUncaughtExceptionHandler;
import org.apache.kafka.streams.processor.AbstractProcessor;
import org.apache.kafka.streams.processor.ProcessorContext;
import org.apache.kafka.streams.processor.To;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.*;

public class UserQueryStreams implements LightStreams {
    static private final Logger logger = LoggerFactory.getLogger(UserQueryStreams.class);
    static private final String APP = "user";
    static final KafkaStreamsConfig streamsConfig = (KafkaStreamsConfig) Config.getInstance().getJsonObjectConfig(KafkaStreamsConfig.CONFIG_NAME, KafkaStreamsConfig.class);
    static final PortalConfig portalConfig = (PortalConfig) Config.getInstance().getJsonObjectConfig(PortalConfig.CONFIG_NAME, PortalConfig.class);
    public static PortalDbProvider dbProvider = (PortalDbProvider) SingletonServiceFactory.getBean(DbProvider.class);

    KafkaStreams userStreams;

    public UserQueryStreams() {
        logger.info("UserQueryStreams is created");
    }


    private void startUserStreams(String ip, int port) {

        final Topology topology = new Topology();
        topology.addSource("SourceTopicProcessor", portalConfig.getTopic());
        topology.addProcessor("UserEventProcessor", UserEventProcessor::new, "SourceTopicProcessor");
        topology.addSink("ScheduleProcessor", portalConfig.getScheduleTopic(), "UserEventProcessor");

        Properties streamsProps = new Properties();
        streamsProps.putAll(streamsConfig.getProperties());
        streamsProps.put(StreamsConfig.DEFAULT_KEY_SERDE_CLASS_CONFIG, Serdes.ByteArray().getClass());
        streamsProps.put(StreamsConfig.DEFAULT_VALUE_SERDE_CLASS_CONFIG, Serdes.ByteArray().getClass());
        streamsProps.put(StreamsConfig.APPLICATION_ID_CONFIG, portalConfig.getUserApplicationId());
        streamsProps.put(StreamsConfig.APPLICATION_SERVER_CONFIG, ip + ":" + port);
        userStreams = new KafkaStreams(topology, streamsProps);
        userStreams.setUncaughtExceptionHandler(ex -> {
            logger.error("Kafka-Streams uncaught exception occurred. Stream will be replaced with new thread", ex);
            return StreamsUncaughtExceptionHandler.StreamThreadExceptionResponse.REPLACE_THREAD;
        });

        if(streamsConfig.isCleanUp()) {
            userStreams.cleanUp();
        }
        userStreams.start();
    }


    public static class UserEventProcessor extends AbstractProcessor<byte[], byte[]> {
        private ProcessorContext pc;

        public UserEventProcessor() {
        }

        @Override
        public void init(ProcessorContext pc) {
            if(logger.isInfoEnabled()) logger.info("Processor initialized");
            this.pc = pc;
        }

        @Override
        public void process(byte[] key, byte[] value) {
            Map<String, Object> event;
            // we need to ignore any message that cannot be deserialized. For example Unknown magic byte!
            try {
                event = Config.getInstance().getMapper().readValue(value, new TypeReference<HashMap<String, Object>>() {});
                String eventType = (String) event.get(CloudEventV1.TYPE);
                Map<String, Object> data = (Map<String, Object>)event.get(PortalConstants.DATA);
                if (logger.isTraceEnabled()) logger.trace("Event type {} event {}", eventType, JsonMapper.toJson(event));
                switch ((String)event.get(CloudEventV1.TYPE)) {
                    // --- User Events ---
                    case PortalConstants.USER_CREATED_EVENT:
                        dbProvider.createUser(event);
                        break;
                    case PortalConstants.SOCIAL_USER_CREATED_EVENT:
                        dbProvider.createSocialUser(event);
                        break;
                    case PortalConstants.USER_CONFIRMED_EVENT:
                        dbProvider.confirmUser(event);
                        break;
                    case PortalConstants.USER_VERIFIED_EVENT:
                        dbProvider.verifyUser(event);
                        break;
                    case PortalConstants.USER_UPDATED_EVENT:
                        dbProvider.updateUser(event);
                        break;
                    case PortalConstants.PASSWORD_FORGOT_EVENT:
                        dbProvider.forgetPassword(event);
                        break;
                    case PortalConstants.PASSWORD_RESET_EVENT:
                        dbProvider.resetPassword(event);
                        break;
                    case PortalConstants.PASSWORD_CHANGED_EVENT:
                        dbProvider.changePassword(event);
                        break;
                    case PortalConstants.USER_DELETED_EVENT:
                        dbProvider.deleteUser(event);
                        break;

                    // --- Org Events ---
                    case PortalConstants.ORG_CREATED_EVENT:
                        dbProvider.createOrg(event);
                        break;
                    case PortalConstants.ORG_UPDATED_EVENT:
                        dbProvider.updateOrg(event);
                        break;
                    case PortalConstants.ORG_DELETED_EVENT:
                        dbProvider.deleteOrg(event);
                        break;

                    // --- Host Events ---
                    case PortalConstants.HOST_CREATED_EVENT:
                        dbProvider.createHost(event);
                        break;
                    case PortalConstants.HOST_UPDATED_EVENT:
                        dbProvider.updateHost(event);
                        break;
                    case PortalConstants.HOST_DELETED_EVENT:
                        dbProvider.deleteHost(event);
                        break;
                    case PortalConstants.HOST_SWITCHED_EVENT:
                        dbProvider.switchHost(event);
                        break;

                    // --- Payment Events ---
                    case PortalConstants.PAYMENT_UPDATED_EVENT:
                        dbProvider.updatePayment(event);
                        break;
                    case PortalConstants.PAYMENT_DELETED_EVENT:
                        dbProvider.deletePayment(event);
                        break;

                    // --- Order Events (Need careful adaptation for Map access) ---
                    case PortalConstants.ORDER_CREATED_EVENT:
                        // handleOrderCreated(event, data); // Extracted complex logic
                        break;
                    case PortalConstants.ORDER_CANCELLED_EVENT:
                        // handleOrderCancelled(event, data); // Extracted complex logic
                        break;
                    case PortalConstants.ORDER_DELIVERED_EVENT:
                        // handleOrderDelivered(event, data); // Extracted complex logic
                        break;

                    // --- Message Events ---
                    case PortalConstants.PRIVATE_MESSAGE_SENT_EVENT:
                        dbProvider.sendPrivateMessage(event);
                        break;

                    // --- Attribute Events ---
                    case PortalConstants.ATTRIBUTE_CREATED_EVENT:
                        dbProvider.createAttribute(event);
                        break;
                    case PortalConstants.ATTRIBUTE_UPDATED_EVENT:
                        dbProvider.updateAttribute(event);
                        break;
                    case PortalConstants.ATTRIBUTE_DELETED_EVENT:
                        dbProvider.deleteAttribute(event);
                        break;
                    case PortalConstants.ATTRIBUTE_PERMISSION_CREATED_EVENT:
                        dbProvider.createAttributePermission(event);
                        break;
                    case PortalConstants.ATTRIBUTE_PERMISSION_UPDATED_EVENT:
                        dbProvider.updateAttributePermission(event);
                        break;
                    case PortalConstants.ATTRIBUTE_PERMISSION_DELETED_EVENT:
                        dbProvider.deleteAttributePermission(event);
                        break;
                    case PortalConstants.ATTRIBUTE_USER_CREATED_EVENT:
                        dbProvider.createAttributeUser(event);
                        break;
                    case PortalConstants.ATTRIBUTE_USER_UPDATED_EVENT:
                        dbProvider.updateAttributeUser(event);
                        break;
                    case PortalConstants.ATTRIBUTE_USER_DELETED_EVENT:
                        dbProvider.deleteAttributeUser(event);
                        break;
                    case PortalConstants.ATTRIBUTE_ROW_FILTER_CREATED_EVENT:
                        dbProvider.createAttributeRowFilter(event);
                        break;
                    case PortalConstants.ATTRIBUTE_ROW_FILTER_UPDATED_EVENT:
                        dbProvider.updateAttributeRowFilter(event);
                        break;
                    case PortalConstants.ATTRIBUTE_ROW_FILTER_DELETED_EVENT:
                        dbProvider.deleteAttributeRowFilter(event);
                        break;
                    case PortalConstants.ATTRIBUTE_COL_FILTER_CREATED_EVENT:
                        dbProvider.createAttributeColFilter(event);
                        break;
                    case PortalConstants.ATTRIBUTE_COL_FILTER_UPDATED_EVENT:
                        dbProvider.updateAttributeColFilter(event);
                        break;
                    case PortalConstants.ATTRIBUTE_COL_FILTER_DELETED_EVENT:
                        dbProvider.deleteAttributeColFilter(event);
                        break;

                    // --- Group Events ---
                    case PortalConstants.GROUP_CREATED_EVENT:
                        dbProvider.createGroup(event);
                        break;
                    case PortalConstants.GROUP_UPDATED_EVENT:
                        dbProvider.updateGroup(event);
                        break;
                    case PortalConstants.GROUP_DELETED_EVENT:
                        dbProvider.deleteGroup(event);
                        break;
                    case PortalConstants.GROUP_PERMISSION_CREATED_EVENT:
                        dbProvider.createGroupPermission(event);
                        break;
                    case PortalConstants.GROUP_PERMISSION_DELETED_EVENT:
                        dbProvider.deleteGroupPermission(event);
                        break;
                    case PortalConstants.GROUP_USER_CREATED_EVENT:
                        dbProvider.createGroupUser(event);
                        break;
                    case PortalConstants.GROUP_USER_UPDATED_EVENT:
                        dbProvider.updateGroupUser(event);
                        break;
                    case PortalConstants.GROUP_USER_DELETED_EVENT:
                        dbProvider.deleteGroupUser(event);
                        break;
                    case PortalConstants.GROUP_ROW_FILTER_CREATED_EVENT:
                        dbProvider.createGroupRowFilter(event);
                        break;
                    case PortalConstants.GROUP_ROW_FILTER_UPDATED_EVENT:
                        dbProvider.updateGroupRowFilter(event);
                        break;
                    case PortalConstants.GROUP_ROW_FILTER_DELETED_EVENT:
                        dbProvider.deleteGroupRowFilter(event);
                        break;
                    case PortalConstants.GROUP_COL_FILTER_CREATED_EVENT:
                        dbProvider.createGroupColFilter(event);
                        break;
                    case PortalConstants.GROUP_COL_FILTER_UPDATED_EVENT:
                        dbProvider.updateGroupColFilter(event);
                        break;
                    case PortalConstants.GROUP_COL_FILTER_DELETED_EVENT:
                        dbProvider.deleteGroupColFilter(event);
                        break;

                    // --- Role Events ---
                    case PortalConstants.ROLE_CREATED_EVENT:
                        dbProvider.createRole(event);
                        break;
                    case PortalConstants.ROLE_UPDATED_EVENT:
                        dbProvider.updateRole(event);
                        break;
                    case PortalConstants.ROLE_DELETED_EVENT:
                        dbProvider.deleteRole(event);
                        break;
                    case PortalConstants.ROLE_PERMISSION_CREATED_EVENT:
                        dbProvider.createRolePermission(event);
                        break;
                    case PortalConstants.ROLE_PERMISSION_DELETED_EVENT:
                        dbProvider.deleteRolePermission(event);
                        break;
                    case PortalConstants.ROLE_USER_CREATED_EVENT:
                        dbProvider.createRoleUser(event);
                        break;
                    case PortalConstants.ROLE_USER_UPDATED_EVENT:
                        dbProvider.updateRoleUser(event);
                        break;
                    case PortalConstants.ROLE_USER_DELETED_EVENT:
                        dbProvider.deleteRoleUser(event);
                        break;
                    case PortalConstants.ROLE_ROW_FILTER_CREATED_EVENT:
                        dbProvider.createRoleRowFilter(event);
                        break;
                    case PortalConstants.ROLE_ROW_FILTER_UPDATED_EVENT:
                        dbProvider.updateRoleRowFilter(event);
                        break;
                    case PortalConstants.ROLE_ROW_FILTER_DELETED_EVENT:
                        dbProvider.deleteRoleRowFilter(event);
                        break;
                    case PortalConstants.ROLE_COL_FILTER_CREATED_EVENT:
                        dbProvider.createRoleColFilter(event);
                        break;
                    case PortalConstants.ROLE_COL_FILTER_UPDATED_EVENT:
                        dbProvider.updateRoleColFilter(event);
                        break;
                    case PortalConstants.ROLE_COL_FILTER_DELETED_EVENT:
                        dbProvider.deleteRoleColFilter(event);
                        break;

                    // --- Position Events ---
                    case PortalConstants.POSITION_CREATED_EVENT:
                        dbProvider.createPosition(event);
                        break;
                    case PortalConstants.POSITION_UPDATED_EVENT:
                        dbProvider.updatePosition(event);
                        break;
                    case PortalConstants.POSITION_DELETED_EVENT:
                        dbProvider.deletePosition(event);
                        break;
                    case PortalConstants.POSITION_PERMISSION_CREATED_EVENT:
                        dbProvider.createPositionPermission(event);
                        break;
                    case PortalConstants.POSITION_PERMISSION_DELETED_EVENT:
                        dbProvider.deletePositionPermission(event);
                        break;
                    case PortalConstants.POSITION_USER_CREATED_EVENT:
                        dbProvider.createPositionUser(event);
                        break;
                    case PortalConstants.POSITION_USER_UPDATED_EVENT:
                        dbProvider.updatePositionUser(event);
                        break;
                    case PortalConstants.POSITION_USER_DELETED_EVENT:
                        dbProvider.deletePositionUser(event);
                        break;
                    case PortalConstants.POSITION_ROW_FILTER_CREATED_EVENT:
                        dbProvider.createPositionRowFilter(event);
                        break;
                    case PortalConstants.POSITION_ROW_FILTER_UPDATED_EVENT:
                        dbProvider.updatePositionRowFilter(event);
                        break;
                    case PortalConstants.POSITION_ROW_FILTER_DELETED_EVENT:
                        dbProvider.deletePositionRowFilter(event);
                        break;
                    case PortalConstants.POSITION_COL_FILTER_CREATED_EVENT:
                        dbProvider.createPositionColFilter(event);
                        break;
                    case PortalConstants.POSITION_COL_FILTER_UPDATED_EVENT:
                        dbProvider.updatePositionColFilter(event);
                        break;
                    case PortalConstants.POSITION_COL_FILTER_DELETED_EVENT:
                        dbProvider.deletePositionColFilter(event);
                        break;

                    // --- Rule Events ---
                    case PortalConstants.RULE_CREATED_EVENT:
                        dbProvider.createRule(event);
                        break;
                    case PortalConstants.RULE_UPDATED_EVENT:
                        dbProvider.updateRule(event);
                        break;
                    case PortalConstants.RULE_DELETED_EVENT:
                        dbProvider.deleteRule(event);
                        break;

                    // --- Schema Events ---
                    case PortalConstants.SCHEMA_CREATED_EVENT:
                        dbProvider.createSchema(event);
                        break;
                    case PortalConstants.SCHEMA_UPDATED_EVENT:
                        dbProvider.updateSchema(event);
                        break;
                    case PortalConstants.SCHEMA_DELETED_EVENT:
                        dbProvider.deleteSchema(event);
                        break;

                    // --- Schedule Events ---
                    case PortalConstants.SCHEDULE_CREATED_EVENT:
                        dbProvider.createSchedule(event);
                        // send the same event schedule created event to the light-scheduler topic for processing
                        // this time, the key is changed to scheduleId which is UUID in order to look it up.
                        pc.forward(((String)data.get("scheduleId")).getBytes(StandardCharsets.UTF_8), value, To.child("ScheduleProcessor"));
                        break;
                    case PortalConstants.SCHEDULE_UPDATED_EVENT:
                        pc.forward(((String)data.get("scheduleId")).getBytes(StandardCharsets.UTF_8), To.child("ScheduleProcessor"));
                        dbProvider.updateSchedule(event);
                        break;
                    case PortalConstants.SCHEDULE_DELETED_EVENT:
                        pc.forward(((String)data.get("scheduleId")).getBytes(StandardCharsets.UTF_8), value, To.child("ScheduleProcessor"));
                        dbProvider.deleteSchedule(event);
                        break;

                    // --- Scheduled Events ---
                    case PortalConstants.PLATFORM_QUERIED_EVENT:
                        // This is the event handler to poll the Ansible or Jenkins deployment platform to check the deployment status.
                        // once the status is completed, it will cancel the schedule, otherwise, it will continue to poll.
                        //dbProvider.updateDeploymentStatus(event); // TODO
                        break;

                    // --- Category Events ---
                    case PortalConstants.CATEGORY_CREATED_EVENT:
                        dbProvider.createCategory(event);
                        break;
                    case PortalConstants.CATEGORY_UPDATED_EVENT:
                        dbProvider.updateCategory(event);
                        break;
                    case PortalConstants.CATEGORY_DELETED_EVENT:
                        dbProvider.deleteCategory(event);
                        break;
                    // --- Tag Events ---
                    case PortalConstants.TAG_CREATED_EVENT:
                        dbProvider.createTag(event);
                        break;
                    case PortalConstants.TAG_UPDATED_EVENT:
                        dbProvider.updateTag(event);
                        break;
                    case PortalConstants.TAG_DELETED_EVENT:
                        dbProvider.deleteTag(event);
                        break;

                    // --- Service Events ---
                    case PortalConstants.SERVICE_CREATED_EVENT:
                        dbProvider.createService(event);
                        break;
                    case PortalConstants.SERVICE_UPDATED_EVENT:
                        dbProvider.updateService(event);
                        break;
                    case PortalConstants.SERVICE_DELETED_EVENT:
                        dbProvider.deleteService(event);
                        break;
                    case PortalConstants.SERVICE_VERSION_CREATED_EVENT:
                        Map<String, Object> map = (Map<String, Object>)event.get(PortalConstants.DATA);
                        String spec = (String)map.get("spec");
                        List<Map<String, Object>> endpoints = null;
                        if(spec != null && !spec.isEmpty()) {
                            endpoints = parseSpec((String)map.get("hostId"),
                                    (String)map.get("apiId"),
                                    (String)map.get("apiVersion"),
                                    (String)map.get("apiType"), spec);
                        }
                        dbProvider.createServiceVersion(event, endpoints);
                        break;
                    case PortalConstants.SERVICE_VERSION_UPDATED_EVENT:
                        map = (Map<String, Object>)event.get(PortalConstants.DATA);
                        spec = (String)map.get("spec");
                        endpoints = null;
                        if(spec != null && !spec.isEmpty()) {
                            endpoints = parseSpec((String)map.get("hostId"),
                                    (String)map.get("apiId"),
                                    (String)map.get("apiVersion"),
                                    (String)map.get("apiType"), spec);
                        }
                        dbProvider.updateServiceVersion(event, endpoints);
                        break;
                    case PortalConstants.SERVICE_VERSION_DELETED_EVENT:
                        dbProvider.deleteServiceVersion(event);
                        break;
                    case PortalConstants.ENDPOINT_RULE_CREATED_EVENT:
                        dbProvider.createEndpointRule(event);
                        break;
                    case PortalConstants.ENDPOINT_RULE_DELETED_EVENT:
                        dbProvider.deleteEndpointRule(event);
                        break;
                    case PortalConstants.SERVICE_SPEC_UPDATED_EVENT:
                        map = (Map<String, Object>)event.get(PortalConstants.DATA);
                        spec = (String)map.get("spec");
                        endpoints = null;
                        if(spec != null && !spec.isEmpty()) {
                            endpoints = parseSpec((String)map.get("hostId"),
                                    (String)map.get("apiId"),
                                    (String)map.get("apiVersion"),
                                    (String)map.get("apiType"), spec);
                        }
                        dbProvider.updateServiceSpec(event, endpoints);
                        break;

                    // --- Auth Events ---
                    case PortalConstants.AUTH_REFRESH_TOKEN_CREATED_EVENT:
                        dbProvider.createRefreshToken(event);
                        break;
                    case PortalConstants.AUTH_REFRESH_TOKEN_DELETED_EVENT:
                        dbProvider.deleteRefreshToken(event);
                        break;
                    case PortalConstants.AUTH_CODE_CREATED_EVENT:
                        dbProvider.createAuthCode(event);
                        break;
                    case PortalConstants.AUTH_CODE_DELETED_EVENT:
                        dbProvider.deleteAuthCode(event);
                        break;
                    case PortalConstants.AUTH_REF_TOKEN_CREATED_EVENT:
                        // dbProvider.createRefToken(event);
                        break;
                    case PortalConstants.AUTH_REF_TOKEN_DELETED_EVENT:
                        // dbProvider.deleteRefToken(event);
                        break;
                    case PortalConstants.AUTH_PROVIDER_CREATED_EVENT:
                        dbProvider.createAuthProvider(event);
                        break;
                    case PortalConstants.AUTH_PROVIDER_ROTATED_EVENT:
                        dbProvider.rotateAuthProvider(event);
                        break;
                    case PortalConstants.AUTH_PROVIDER_UPDATED_EVENT:
                        dbProvider.updateAuthProvider(event);
                        break;
                    case PortalConstants.AUTH_PROVIDER_DELETED_EVENT:
                        dbProvider.deleteAuthProvider(event);
                        break;

                    // --- Product Events ---
                    case PortalConstants.PRODUCT_CREATED_EVENT:
                        dbProvider.createProduct(event);
                        break;
                    case PortalConstants.PRODUCT_UPDATED_EVENT:
                        dbProvider.updateProduct(event);
                        break;
                    case PortalConstants.PRODUCT_DELETED_EVENT:
                        dbProvider.deleteProduct(event);
                        break;

                    // --- Product Environment Events ---
                    case PortalConstants.PRODUCT_VERSION_ENVIRONMENT_CREATED_EVENT:
                        dbProvider.createProductVersionEnvironment(event);
                        break;
                    case PortalConstants.PRODUCT_VERSION_ENVIRONMENT_DELETED_EVENT:
                        dbProvider.deleteProductVersionEnvironment(event);
                        break;

                    // --- Product Pipeline Events ---
                    case PortalConstants.PRODUCT_VERSION_PIPELINE_CREATED_EVENT:
                        dbProvider.createProductVersionPipeline(event);
                        break;
                    case PortalConstants.PRODUCT_VERSION_PIPELINE_DELETED_EVENT:
                        dbProvider.deleteProductVersionPipeline(event);
                        break;

                    // --- Product Version Config Events ---
                    case PortalConstants.PRODUCT_VERSION_CONFIG_CREATED_EVENT:
                        dbProvider.createProductVersionConfig(event);
                        break;
                    case PortalConstants.PRODUCT_VERSION_CONFIG_DELETED_EVENT:
                        dbProvider.deleteProductVersionConfig(event);
                        break;

                    // --- Product Version Config Property Events ---
                    case PortalConstants.PRODUCT_VERSION_CONFIG_PROPERTY_CREATED_EVENT:
                        dbProvider.createProductVersionConfigProperty(event);
                        break;
                    case PortalConstants.PRODUCT_VERSION_CONFIG_PROPERTY_DELETED_EVENT:
                        dbProvider.deleteProductVersionConfigProperty(event);
                        break;

                    // --- Pipeline Events ---
                    case PortalConstants.PIPELINE_CREATED_EVENT:
                        dbProvider.createPipeline(event);
                        break;
                    case PortalConstants.PIPELINE_UPDATED_EVENT:
                        dbProvider.updatePipeline(event);
                        break;
                    case PortalConstants.PIPELINE_DELETED_EVENT:
                        dbProvider.deletePipeline(event);
                        break;

                    // --- Platform Events ---
                    case PortalConstants.PLATFORM_CREATED_EVENT:
                        dbProvider.createPlatform(event);
                        break;
                    case PortalConstants.PLATFORM_UPDATED_EVENT:
                        dbProvider.updatePlatform(event);
                        break;
                    case PortalConstants.PLATFORM_DELETED_EVENT:
                        dbProvider.deletePlatform(event);
                        break;

                    // --- Instance Events ---
                    case PortalConstants.INSTANCE_CREATED_EVENT:
                        dbProvider.createInstance(event);
                        break;
                    case PortalConstants.INSTANCE_UPDATED_EVENT:
                        dbProvider.updateInstance(event);
                        break;
                    case PortalConstants.INSTANCE_DELETED_EVENT:
                        dbProvider.deleteInstance(event);
                        break;
                    case PortalConstants.INSTANCE_API_CREATED_EVENT:
                        dbProvider.createInstanceApi(event);
                        break;
                    case PortalConstants.INSTANCE_API_UPDATED_EVENT:
                        dbProvider.updateInstanceApi(event);
                        break;
                    case PortalConstants.INSTANCE_API_DELETED_EVENT:
                        dbProvider.deleteInstanceApi(event);
                        break;
                    case PortalConstants.INSTANCE_APP_CREATED_EVENT:
                        dbProvider.createInstanceApp(event);
                        break;
                    case PortalConstants.INSTANCE_APP_UPDATED_EVENT:
                        dbProvider.updateInstanceApp(event);
                        break;
                    case PortalConstants.INSTANCE_APP_DELETED_EVENT:
                        dbProvider.deleteInstanceApp(event);
                        break;
                    case PortalConstants.INSTANCE_APP_API_CREATED_EVENT:
                        dbProvider.createInstanceAppApi(event);
                        break;
                    case PortalConstants.INSTANCE_APP_API_UPDATED_EVENT:
                        dbProvider.updateInstanceAppApi(event);
                        break;
                    case PortalConstants.INSTANCE_APP_API_DELETED_EVENT:
                        dbProvider.deleteInstanceAppApi(event);
                        break;
                    case PortalConstants.INSTANCE_API_PATH_PREFIX_CREATED_EVENT:
                        dbProvider.createInstanceApiPathPrefix(event);
                        break;
                    case PortalConstants.INSTANCE_API_PATH_PREFIX_UPDATED_EVENT:
                        dbProvider.updateInstanceApiPathPrefix(event);
                        break;
                    case PortalConstants.INSTANCE_API_PATH_PREFIX_DELETED_EVENT:
                        dbProvider.deleteInstanceApiPathPrefix(event);
                        break;

                    // --- Deployment Events ---
                    case PortalConstants.DEPLOYMENT_CREATED_EVENT:
                        dbProvider.createDeployment(event);
                        break;
                    case PortalConstants.DEPLOYMENT_UPDATED_EVENT:
                        dbProvider.updateDeployment(event);
                        break;
                    case PortalConstants.DEPLOYMENT_JOB_ID_UPDATED_EVENT:
                        dbProvider.updateDeploymentJobId(event);
                        break;
                    case PortalConstants.DEPLOYMENT_STATUS_UPDATED_EVENT:
                        dbProvider.updateDeploymentStatus(event);
                        break;
                    case PortalConstants.DEPLOYMENT_DELETED_EVENT:
                        dbProvider.deleteDeployment(event);
                        break;

                    // --- Deployment Instance Events ---
                    case PortalConstants.DEPLOYMENT_INSTANCE_CREATED_EVENT:
                        dbProvider.createDeploymentInstance(event);
                        break;
                    case PortalConstants.DEPLOYMENT_INSTANCE_UPDATED_EVENT:
                        dbProvider.updateDeploymentInstance(event);
                        break;
                    case PortalConstants.DEPLOYMENT_INSTANCE_DELETED_EVENT:
                        dbProvider.deleteDeploymentInstance(event);
                        break;

                    // --- Config Events ---
                    case PortalConstants.CONFIG_CREATED_EVENT:
                        dbProvider.createConfig(event);
                        break;
                    case PortalConstants.CONFIG_UPDATED_EVENT:
                        dbProvider.updateConfig(event);
                        break;
                    case PortalConstants.CONFIG_DELETED_EVENT:
                        dbProvider.deleteConfig(event);
                        break;
                    case PortalConstants.CONFIG_PROPERTY_CREATED_EVENT:
                        dbProvider.createConfigProperty(event);
                        break;
                    case PortalConstants.CONFIG_PROPERTY_UPDATED_EVENT:
                        dbProvider.updateConfigProperty(event);
                        break;
                    case PortalConstants.CONFIG_PROPERTY_DELETED_EVENT:
                        dbProvider.deleteConfigProperty(event);
                        break;
                    case PortalConstants.CONFIG_ENVIRONMENT_CREATED_EVENT:
                        dbProvider.createConfigEnvironment(event);
                        break;
                    case PortalConstants.CONFIG_ENVIRONMENT_UPDATED_EVENT:
                        dbProvider.updateConfigEnvironment(event);
                        break;
                    case PortalConstants.CONFIG_ENVIRONMENT_DELETED_EVENT:
                        dbProvider.deleteConfigEnvironment(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_API_CREATED_EVENT:
                        dbProvider.createConfigInstanceApi(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_API_UPDATED_EVENT:
                        dbProvider.updateConfigInstanceApi(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_API_DELETED_EVENT:
                        dbProvider.deleteConfigInstanceApi(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_APP_CREATED_EVENT:
                        dbProvider.createConfigInstanceApp(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_APP_UPDATED_EVENT:
                        dbProvider.updateConfigInstanceApp(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_APP_DELETED_EVENT:
                        dbProvider.deleteConfigInstanceApp(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_APP_API_CREATED_EVENT:
                        dbProvider.createConfigInstanceAppApi(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_APP_API_UPDATED_EVENT:
                        dbProvider.updateConfigInstanceAppApi(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_APP_API_DELETED_EVENT:
                        dbProvider.deleteConfigInstanceAppApi(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_FILE_CREATED_EVENT:
                        dbProvider.createConfigInstanceFile(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_FILE_UPDATED_EVENT:
                        dbProvider.updateConfigInstanceFile(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_FILE_DELETED_EVENT:
                        dbProvider.deleteConfigInstanceFile(event);
                        break;
                    case PortalConstants.CONFIG_DEPLOYMENT_INSTANCE_CREATED_EVENT:
                        dbProvider.createConfigDeploymentInstance(event);
                        break;
                    case PortalConstants.CONFIG_DEPLOYMENT_INSTANCE_UPDATED_EVENT:
                        dbProvider.updateConfigDeploymentInstance(event);
                        break;
                    case PortalConstants.CONFIG_DEPLOYMENT_INSTANCE_DELETED_EVENT:
                        dbProvider.deleteConfigDeploymentInstance(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_CREATED_EVENT:
                        dbProvider.createConfigInstance(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_UPDATED_EVENT:
                        dbProvider.updateConfigInstance(event);
                        break;
                    case PortalConstants.CONFIG_INSTANCE_DELETED_EVENT:
                        dbProvider.deleteConfigInstance(event);
                        break;
                    case PortalConstants.CONFIG_PRODUCT_CREATED_EVENT:
                        dbProvider.createConfigProduct(event);
                        break;
                    case PortalConstants.CONFIG_PRODUCT_UPDATED_EVENT:
                        dbProvider.updateConfigProduct(event);
                        break;
                    case PortalConstants.CONFIG_PRODUCT_DELETED_EVENT:
                        dbProvider.deleteConfigProduct(event);
                        break;
                    case PortalConstants.CONFIG_PRODUCT_VERSION_CREATED_EVENT:
                        dbProvider.createConfigProductVersion(event);
                        break;
                    case PortalConstants.CONFIG_PRODUCT_VERSION_UPDATED_EVENT:
                        dbProvider.updateConfigProductVersion(event);
                        break;
                    case PortalConstants.CONFIG_PRODUCT_VERSION_DELETED_EVENT:
                        dbProvider.deleteConfigProductVersion(event);
                        break;

                    // --- App Events ---
                    case PortalConstants.APP_CREATED_EVENT:
                        dbProvider.createApp(event);
                        break;
                    case PortalConstants.APP_UPDATED_EVENT:
                        dbProvider.updateApp(event);
                        break;
                    case PortalConstants.APP_DELETED_EVENT:
                        dbProvider.deleteApp(event);
                        break;

                    // --- Client Events ---
                    case PortalConstants.CLIENT_CREATED_EVENT:
                        dbProvider.createClient(event);
                        break;
                    case PortalConstants.CLIENT_UPDATED_EVENT:
                        dbProvider.updateClient(event);
                        break;
                    case PortalConstants.CLIENT_DELETED_EVENT:
                        dbProvider.deleteClient(event);
                        break;
                    // --- Reference Table Events ---
                    case PortalConstants.REF_TABLE_CREATED_EVENT:
                        dbProvider.createRefTable(event);
                        break;
                    case PortalConstants.REF_TABLE_UPDATED_EVENT:
                        dbProvider.updateRefTable(event);
                        break;
                    case PortalConstants.REF_TABLE_DELETED_EVENT:
                        dbProvider.deleteRefTable(event);
                        break;
                    case PortalConstants.REF_VALUE_CREATED_EVENT:
                        dbProvider.createRefValue(event);
                        break;
                    case PortalConstants.REF_VALUE_UPDATED_EVENT:
                        dbProvider.updateRefValue(event);
                        break;
                    case PortalConstants.REF_VALUE_DELETED_EVENT:
                        dbProvider.deleteRefValue(event);
                        break;
                    case PortalConstants.REF_LOCALE_CREATED_EVENT:
                        dbProvider.createRefLocale(event);
                        break;
                    case PortalConstants.REF_LOCALE_UPDATED_EVENT:
                        dbProvider.updateRefLocale(event);
                        break;
                    case PortalConstants.REF_LOCALE_DELETED_EVENT:
                        dbProvider.deleteRefLocale(event);
                        break;
                    case PortalConstants.REF_RELATION_TYPE_CREATED_EVENT:
                        dbProvider.createRefRelationType(event);
                        break;
                    case PortalConstants.REF_RELATION_TYPE_UPDATED_EVENT:
                        dbProvider.updateRefRelationType(event);
                        break;
                    case PortalConstants.REF_RELATION_TYPE_DELETED_EVENT:
                        dbProvider.deleteRefRelationType(event);
                        break;
                    case PortalConstants.REF_RELATION_CREATED_EVENT:
                        dbProvider.createRefRelation(event);
                        break;
                    case PortalConstants.REF_RELATION_UPDATED_EVENT:
                        dbProvider.updateRefRelation(event);
                        break;
                    case PortalConstants.REF_RELATION_DELETED_EVENT:
                        dbProvider.deleteRefRelation(event);
                        break;

                    // --- Default Case ---
                    default:
                        logger.error("Unhandled event type '{}': {}", eventType, event);
                        // Optional: Send to DLQ or handle differently
                        break;
                }
            } catch (Exception e) {
                logger.error("Exception:", e);
            }
            /*
            try {
                if(event.getType() instanceof UserCreatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("UserCreatedEvent = {}", event);
                    dbProvider.createUser(event);
                } else if(object instanceof SocialUserCreatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("SocialUserCreatedEvent = {}", event);
                    dbProvider.createSocialUser(event);
                } else if(object instanceof UserConfirmedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("UserConfirmedEvent = {}", event);
                    dbProvider.confirmUser(event);
                } else if(object instanceof UserVerifiedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("UserVerifiedEvent = {}", event);
                    dbProvider.verifyUser(event);
                } else if(object instanceof UserUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("UserUpdatedEvent = {}", event);
                    dbProvider.updateUser(event);
                } else if(object instanceof PasswordForgotEvent event) {
                    // make sure that we have user available.
                    if(logger.isTraceEnabled()) logger.trace("PasswordForgotEvent = {}", event);
                    dbProvider.forgetPassword(event);
                } else if(object instanceof PasswordResetEvent event) {
                    // make sure that we have user available.
                    if(logger.isTraceEnabled()) logger.trace("PasswordResetEvent = {}", event);
                    dbProvider.resetPassword(event);
                } else if(object instanceof PasswordChangedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PasswordChangedEvent = {}", event);
                    dbProvider.changePassword(event);
                } else if(object instanceof UserDeletedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("UserDeletedEvent = {}", event);
                    dbProvider.deleteUser(event);
                } else if(object instanceof UserRolesUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("UserRolesUpdatedEvent = {}", event);
                    dbProvider.updateUserRoles(event);
                } else if(object instanceof OrgCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("OrgCreatedEvent = {}", event);
                    dbProvider.createOrg(event);
                } else if(object instanceof OrgUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("OrgUpdatedEvent = {}", event);
                    dbProvider.updateOrg(event);
                } else if(object instanceof OrgDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("OrgDeletedEvent = {}", event);
                    dbProvider.deleteOrg(event);
                } else if(object instanceof HostCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("HostCreatedEvent = {}", event);
                    dbProvider.createHost(event);
                } else if(object instanceof HostUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("HostUpdatedEvent = {}", event);
                    dbProvider.updateHost(event);
                } else if(object instanceof HostDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("HostDeletedEvent = {}", event);
                    dbProvider.deleteHost(event);
                } else if(object instanceof HostSwitchedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("HostSwitchedEvent = {}", event);
                    dbProvider.switchHost(event);
                } else if(object instanceof PaymentUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PaymentUpdatedEvent = {}", event);
                    dbProvider.updatePayment(event);
                } else if(object instanceof PaymentDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PaymentDeletedEvent = {}", event);
                    dbProvider.deletePayment(event);
                } else if(object instanceof OrderCreatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("OrderCreatedEvent = {}", event);
                    dbProvider.createOrder(event);

                    String id = orderCreatedEvent.getEventId().getId();
                    long nonce = orderCreatedEvent.getEventId().getNonce();
                    String email = orderCreatedEvent.getEmail();
                    String order = orderCreatedEvent.getOrder();
                    if (id.equals(email)) {
                        // customer order flattened structure as there should not be too many
                        String orderString = customerOrderStore.get(email);
                        if (orderString == null) {
                            Map<String, Object> orderMap = JsonMapper.string2Map(order);
                            List<Map<String, Object>> orders = new ArrayList<>();
                            orders.add(orderMap);
                            customerOrderStore.put(email, JsonMapper.toJson(orders));
                        } else {
                            Map<String, Object> orderMap = JsonMapper.string2Map(order);
                            List<Map<String, Object>> orders = JsonMapper.string2List(orderString);
                            orders.add(orderMap);
                            customerOrderStore.put(email, JsonMapper.toJson(orders));
                        }
                    } else {
                        // merchant order based on the order status: CONFIRMED, CANCELLED, DELIVERED
                        // all newly created orders are in the Confirmed map initially.
                        String orderString = merchantOrderStore.get(email);
                        if (orderString == null) {
                            Map<String, List> orderMap = new HashMap<>();
                            List<Map<String, Object>> confirmedOrders = new ArrayList<>();
                            confirmedOrders.add(JsonMapper.string2Map(order));
                            orderMap.put(PortalConstants.ORDER_CONFIRMED, confirmedOrders);
                            merchantOrderStore.put(email, JsonMapper.toJson(orderMap));
                        } else {
                            Map<String, Object> orderMap = JsonMapper.string2Map(orderString);
                            List<Map<String, Object>> confirmedOrders = (List) orderMap.get(PortalConstants.ORDER_CONFIRMED);
                            if (confirmedOrders == null) {
                                confirmedOrders = new ArrayList<>();
                                confirmedOrders.add(JsonMapper.string2Map(order));
                                orderMap.put(PortalConstants.ORDER_CONFIRMED, confirmedOrders);
                            } else {
                                confirmedOrders.add(JsonMapper.string2Map(order));
                            }
                            merchantOrderStore.put(email, JsonMapper.toJson(orderMap));
                        }
                    }
                    pc.forward(id.getBytes(StandardCharsets.UTF_8), ByteUtil.longToBytes(nonce + 1), To.child("NonceProcessor"));
                    EventNotification notification = new EventNotification(nonce, APP, orderCreatedEvent.getClass().getSimpleName(), true, null, orderCreatedEvent);
                    pc.forward(id.getBytes(StandardCharsets.UTF_8), notification.toString().getBytes(StandardCharsets.UTF_8), To.child("NotificationProcessor"));
                } else if(object instanceof OrderCancelledEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("OrderCancelledEvent = {}", event);
                    dbProvider.cancelOrder(event);
                    String id = orderCancelledEvent.getEventId().getId();
                    long nonce = orderCancelledEvent.getEventId().getNonce();
                    String email = orderCancelledEvent.getEmail();
                    String merchantEmail = orderCancelledEvent.getMerchantEmail();
                    String orderId = orderCancelledEvent.getOrderId();
                    // update the customer order store
                    String customerOrderString = customerOrderStore.get(email);
                    if (customerOrderString != null) {
                        List<Map<String, Object>> customerOrders = JsonMapper.string2List(customerOrderString);
                        // find the orderId and update the status there.
                        for(Map<String, Object> order: customerOrders) {
                            if(orderId.equals(order.get("orderId"))) {
                                order.put("status", PortalConstants.ORDER_CANCELLED);
                                break;
                            }
                        }
                        customerOrderStore.put(email, JsonMapper.toJson(customerOrders));
                    }
                    String merchantOrderString = merchantOrderStore.get(merchantEmail);
                    if(merchantOrderString != null) {
                        Map<String, Object> orderMap = JsonMapper.string2Map(merchantOrderString);
                        List<Map<String, Object>> confirmedOrders = (List)orderMap.get(PortalConstants.ORDER_CONFIRMED);
                        Map<String, Object> removed = null;
                        Iterator<Map<String, Object>> iterator = confirmedOrders.iterator();
                        while(iterator.hasNext()) {
                            Map<String, Object> order = iterator.next();
                            if(orderId.equals(order.get("orderId"))) {
                                removed = order;
                                iterator.remove();
                            }
                        }
                        removed.put("status", PortalConstants.ORDER_CANCELLED);
                        List<Map<String, Object>> cancelledOrders = (List)orderMap.get(PortalConstants.ORDER_CANCELLED);
                        if(cancelledOrders == null) {
                            cancelledOrders = new ArrayList<>();
                            orderMap.put(PortalConstants.ORDER_CANCELLED, cancelledOrders);
                        }
                        cancelledOrders.add(removed);
                        merchantOrderStore.put(merchantEmail, JsonMapper.toJson(orderMap));
                    }
                    pc.forward(id.getBytes(StandardCharsets.UTF_8), ByteUtil.longToBytes(nonce + 1), To.child("NonceProcessor"));
                    EventNotification notification = new EventNotification(nonce, APP, orderCancelledEvent.getClass().getSimpleName(), true, null, orderCancelledEvent);
                    pc.forward(id.getBytes(StandardCharsets.UTF_8), notification.toString().getBytes(StandardCharsets.UTF_8), To.child("NotificationProcessor"));
                } else if(object instanceof OrderDeliveredEvent event) {
                    // The status updates happen on two partitions for both customer order and merchant order.
                    if (logger.isTraceEnabled()) logger.trace("OrderDeliveredEvent = {}", event);
                    dbProvider.deliverOrder(event);
                    String id = orderDeliveredEvent.getEventId().getId();
                    long nonce = orderDeliveredEvent.getEventId().getNonce();
                    String email = orderDeliveredEvent.getEmail();
                    String customerEmail = orderDeliveredEvent.getCustomerEmail();
                    String orderId = orderDeliveredEvent.getOrderId();

                    // update the customer order store
                    String customerOrderString = customerOrderStore.get(customerEmail);
                    if (customerOrderString != null) {
                        List<Map<String, Object>> customerOrders = JsonMapper.string2List(customerOrderString);
                        // find the orderId and update the status there.
                        for(Map<String, Object> order: customerOrders) {
                            if(orderId.equals(order.get("orderId"))) {
                                order.put("status", PortalConstants.ORDER_DELIVERED);
                                break;
                            }
                        }
                        customerOrderStore.put(customerEmail, JsonMapper.toJson(customerOrders));
                    }
                    String merchantOrderString = merchantOrderStore.get(email);
                    if(merchantOrderString != null) {
                        Map<String, Object> orderMap = JsonMapper.string2Map(merchantOrderString);
                        List<Map<String, Object>> confirmedOrders = (List)orderMap.get(PortalConstants.ORDER_CONFIRMED);
                        Map<String, Object> removed = null;
                        Iterator<Map<String, Object>> iterator = confirmedOrders.iterator();
                        while(iterator.hasNext()) {
                            Map<String, Object> order = iterator.next();
                            if(orderId.equals(order.get("orderId"))) {
                                removed = order;
                                iterator.remove();
                            }
                        }
                        removed.put("status", PortalConstants.ORDER_DELIVERED);
                        List<Map<String, Object>> deliveredOrders = (List)orderMap.get(PortalConstants.ORDER_DELIVERED);
                        if(deliveredOrders == null) {
                            deliveredOrders = new ArrayList<>();
                            orderMap.put(PortalConstants.ORDER_DELIVERED, deliveredOrders);
                        }
                        deliveredOrders.add(removed);
                        merchantOrderStore.put(email, JsonMapper.toJson(orderMap));
                    }

                    pc.forward(id.getBytes(StandardCharsets.UTF_8), ByteUtil.longToBytes(nonce + 1), To.child("NonceProcessor"));
                    EventNotification notification = new EventNotification(nonce, APP, orderDeliveredEvent.getClass().getSimpleName(), true, null, orderDeliveredEvent);
                    pc.forward(id.getBytes(StandardCharsets.UTF_8), notification.toString().getBytes(StandardCharsets.UTF_8), To.child("NotificationProcessor"));
                } else if(object instanceof ReferenceCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ReferenceCreatedEvent = {}", event);
                    // TODO these events should be rewritten to update reference tables in the database.
                } else if(object instanceof ReferenceUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ReferenceUpdatedEvent = {}", event);
                    // TODO these events should be rewritten to update reference tables in the database.
                } else if(object instanceof ReferenceDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ReferenceDeletedEvent = {}", event);
                    // TODO these events should be rewritten to update reference tables in the database.
                } else if(object instanceof PrivateMessageSentEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PrivateMessageSentEvent = {}", event);
                    dbProvider.sendPrivateMessage(event);
                } else if(object instanceof AttributeCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeCreatedEvent = {}", event);
                    dbProvider.createAttribute(event);
                } else if(object instanceof AttributeUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeUpdatedEvent = {}", event);
                    dbProvider.updateAttribute(event);
                } else if(object instanceof AttributeDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeDeletedEvent = {}", event);
                    dbProvider.deleteAttribute(event);
                } else if(object instanceof AttributePermissionCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributePermissionCreatedEvent = {}", event);
                    dbProvider.createAttributePermission(event);
                } else if(object instanceof AttributePermissionUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributePermissionUpdatedEvent = {}", event);
                    dbProvider.updateAttributePermission(event);
                } else if(object instanceof AttributePermissionDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributePermissionDeletedEvent = {}", event);
                    dbProvider.deleteAttributePermission(event);
                } else if(object instanceof AttributeUserCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeUserCreatedEvent = {}", event);
                    dbProvider.createAttributeUser(event);
                } else if(object instanceof AttributeUserUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeUserUpdatedEvent = {}", event);
                    dbProvider.updateAttributeUser(event);
                } else if(object instanceof AttributeUserDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeUserDeletedEvent = {}", event);
                    dbProvider.deleteAttributeUser(event);
                } else if(object instanceof AttributeRowFilterCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeRowFilterCreatedEvent = {}", event);
                    dbProvider.createAttributeRowFilter(event);
                } else if(object instanceof AttributeRowFilterUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeRowFilterUpdatedEvent = {}", event);
                    dbProvider.updateAttributeRowFilter(event);
                } else if(object instanceof AttributeRowFilterDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeRowFilterDeletedEvent = {}", event);
                    dbProvider.deleteAttributeRowFilter(event);
                } else if(object instanceof AttributeColFilterCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeColFilterCreatedEvent = {}", event);
                    dbProvider.createAttributeColFilter(event);
                } else if(object instanceof AttributeColFilterUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeColFilterUpdatedEvent = {}", event);
                    dbProvider.updateAttributeColFilter(event);
                } else if(object instanceof AttributeColFilterDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AttributeColFilterDeletedEvent = {}", event);
                    dbProvider.deleteAttributeColFilter(event);
                } else if(object instanceof GroupCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupCreatedEvent = {}", event);
                    dbProvider.createGroup(event);
                } else if(object instanceof GroupUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupUpdatedEvent = {}", event);
                    dbProvider.updateGroup(event);
                } else if(object instanceof GroupDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupDeletedEvent = {}", event);
                    dbProvider.deleteGroup(event);
                } else if(object instanceof GroupPermissionCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupPermissionCreatedEvent = {}", event);
                    dbProvider.createGroupPermission(event);
                } else if(object instanceof GroupPermissionDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupPermissionDeletedEvent = {}", event);
                    dbProvider.deleteGroupPermission(event);
                } else if(object instanceof GroupUserCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupUserCreatedEvent = {}", event);
                    dbProvider.createGroupUser(event);
                } else if(object instanceof GroupUserUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupUserUpdatedEvent = {}", event);
                    dbProvider.updateGroupUser(event);
                } else if(object instanceof GroupUserDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupUserDeletedEvent = {}", event);
                    dbProvider.deleteGroupUser(event);
                } else if(object instanceof GroupRowFilterCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupRowFilterCreatedEvent = {}", event);
                    dbProvider.createGroupRowFilter(event);
                } else if(object instanceof GroupRowFilterUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupRowFilterUpdatedEvent = {}", event);
                    dbProvider.updateGroupRowFilter(event);
                } else if(object instanceof GroupRowFilterDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupRowFilterDeletedEvent = {}", event);
                    dbProvider.deleteGroupRowFilter(event);
                } else if(object instanceof GroupColFilterCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupColFilterCreatedEvent = {}", event);
                    dbProvider.createGroupColFilter(event);
                } else if(object instanceof GroupColFilterUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupColFilterUpdatedEvent = {}", event);
                    dbProvider.updateGroupColFilter(event);
                } else if(object instanceof GroupColFilterDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("GroupColFilterDeletedEvent = {}", event);
                    dbProvider.deleteGroupColFilter(event);
                } else if(object instanceof RoleCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleCreatedEvent = {}", event);
                    dbProvider.createRole(event);
                } else if(object instanceof RoleUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleUpdatedEvent = {}", event);
                    dbProvider.updateRole(event);
                } else if(object instanceof RoleDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleDeletedEvent = {}", event);
                    dbProvider.deleteRole(event);
                } else if(object instanceof RolePermissionCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RolePermissionCreatedEvent = {}", event);
                    dbProvider.createRolePermission(event);
                } else if(object instanceof RolePermissionDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RolePermissionDeletedEvent = {}", event);
                    dbProvider.deleteRolePermission(event);
                } else if(object instanceof RoleUserCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleUserCreatedEvent = {}", event);
                    dbProvider.createRoleUser(event);
                } else if(object instanceof RoleUserUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleUserUpdatedEvent = {}", event);
                    dbProvider.updateRoleUser(event);
                } else if(object instanceof RoleUserDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleUserDeletedEvent = {}", event);
                    dbProvider.deleteRoleUser(event);
                } else if(object instanceof RoleRowFilterCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleRowFilterCreatedEvent = {}", event);
                    dbProvider.createRoleRowFilter(event);
                } else if(object instanceof RoleRowFilterUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleRowFilterUpdatedEvent = {}", event);
                    dbProvider.updateRoleRowFilter(event);
                } else if(object instanceof RoleRowFilterDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleRowFilterDeletedEvent = {}", event);
                    dbProvider.deleteRoleRowFilter(event);
                } else if(object instanceof RoleColFilterCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleColFilterCreatedEvent = {}", event);
                    dbProvider.createRoleColFilter(event);
                } else if(object instanceof RoleColFilterUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleColFilterUpdatedEvent = {}", event);
                    dbProvider.updateRoleColFilter(event);
                } else if(object instanceof RoleColFilterDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("RoleColFilterDeletedEvent = {}", event);
                    dbProvider.deleteRoleColFilter(event);
                } else if(object instanceof PositionCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionCreatedEvent = {}", event);
                    dbProvider.createPosition(event);
                } else if(object instanceof PositionUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionUpdatedEvent = {}", event);
                    dbProvider.updatePosition(event);
                } else if(object instanceof PositionDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionDeletedEvent = {}", event);
                    dbProvider.deletePosition(event);
                } else if(object instanceof PositionPermissionCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionPermissionCreatedEvent = {}", event);
                    dbProvider.createPositionPermission(event);
                } else if(object instanceof PositionPermissionDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionPermissionDeletedEvent = {}", event);
                    dbProvider.deletePositionPermission(event);
                } else if(object instanceof PositionUserCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionUserCreatedEvent = {}", event);
                    dbProvider.createPositionUser(event);
                } else if(object instanceof PositionUserUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionUserUpdatedEvent = {}", event);
                    dbProvider.updatePositionUser(event);
                } else if(object instanceof PositionUserDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionUserDeletedEvent = {}", event);
                    dbProvider.deletePositionUser(event);
                } else if(object instanceof PositionRowFilterCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionRowFilterCreatedEvent = {}", event);
                    dbProvider.createPositionRowFilter(event);
                } else if(object instanceof PositionRowFilterUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionRowFilterUpdatedEvent = {}", event);
                    dbProvider.updatePositionRowFilter(event);
                } else if(object instanceof PositionRowFilterDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionRowFilterDeletedEvent = {}", event);
                    dbProvider.deletePositionRowFilter(event);
                } else if(object instanceof PositionColFilterCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionColFilterCreatedEvent = {}", event);
                    dbProvider.createPositionColFilter(event);
                } else if(object instanceof PositionColFilterUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionColFilterUpdatedEvent = {}", event);
                    dbProvider.updatePositionColFilter(event);
                } else if(object instanceof PositionColFilterDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PositionColFilterDeletedEvent = {}", event);
                    dbProvider.deletePositionColFilter(event);
                } else if(object instanceof RuleCreatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("RuleCreatedEvent = {}", event);
                    dbProvider.createRule(event);
                } else if(object instanceof RuleUpdatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("RuleUpdatedEvent = {}", event);
                    dbProvider.updateRule(event);
                } else if(object instanceof RuleDeletedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("RuleDeletedEvent = {}", event);
                    dbProvider.deleteRule(event);
                } else if(object instanceof ServiceCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ServiceCreatedEvent = {}", event);
                    dbProvider.createService(event);
                } else if(object instanceof ServiceUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ServiceUpdatedEvent = {}", event);
                    dbProvider.updateService(event);
                } else if(object instanceof ServiceSpecUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ServiceSpecUpdatedEvent = {}", event);
                    // based on the apiType, parse the spec into endpoints and save it into the service store
                    String apiId = event.getApiId();
                    String hostId = event.getHostId();
                    String apiVersion = event.getApiVersion();
                    String apiType = event.getApiType();
                    String spec = event.getSpec().trim();
                    List<Map<String, Object>> endpoints = null;
                    if(spec != null && spec.length() > 0) {
                        endpoints = parseSpec(hostId, apiId, apiVersion, apiType, spec);
                    }
                    dbProvider.updateServiceSpec(event, endpoints);
                } else if(object instanceof ServiceDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ServiceDeletedEvent = {}", event);
                    dbProvider.deleteService(event);
                } else if(object instanceof ServiceVersionCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ServiceVersionCreatedEvent = {}", event);
                    String apiId = event.getApiId();
                    String hostId = event.getHostId();
                    String apiVersion = event.getApiVersion();
                    String apiType = event.getApiType();
                    Map<String, Object> valueMap = JsonMapper.string2Map(event.getValue());
                    String spec = (String)valueMap.get("spec");
                    List<Map<String, Object>> endpoints = null;
                    if(spec != null && spec.length() > 0) {
                        endpoints = parseSpec(hostId, apiId, apiVersion, apiType, spec);
                    }
                    dbProvider.createServiceVersion(event, endpoints);
                } else if(object instanceof ServiceVersionUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ServiceVersionUpdatedEvent = {}", event);
                    String apiId = event.getApiId();
                    String hostId = event.getHostId();
                    String apiVersion = event.getApiVersion();
                    String apiType = event.getApiType();
                    Map<String, Object> valueMap = JsonMapper.string2Map(event.getValue());
                    String spec = (String)valueMap.get("spec");
                    List<Map<String, Object>> endpoints = null;
                    if(spec != null && spec.length() > 0) {
                        endpoints = parseSpec(hostId, apiId, apiVersion, apiType, spec);
                    }
                    dbProvider.updateServiceVersion(event, endpoints);
                } else if(object instanceof ServiceVersionDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ServiceVersionDeletedEvent = {}", event);
                    dbProvider.deleteServiceVersion(event);
                } else if(object instanceof EndpointRuleCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("EndpointRuleCreatedEvent = {}", event);
                    dbProvider.createEndpointRule(event);
                } else if(object instanceof EndpointRuleDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("EndpointRuleDeletedEvent = {}", event);
                    dbProvider.deleteEndpointRule(event);
                } else if(object instanceof AuthRefreshTokenCreatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthRefreshTokenCreatedEvent = {}", event);
                    dbProvider.createRefreshToken(event);
                } else if(object instanceof AuthRefreshTokenDeletedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthRefreshTokenDeletedEvent = {}", event);
                    dbProvider.deleteRefreshToken(event);
                } else if(object instanceof AuthCodeCreatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthCodeCreatedEvent = {}", event);
                    dbProvider.createAuthCode(event);
                } else if(object instanceof AuthCodeDeletedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthCodeDeletedEvent = {}", event);
                    dbProvider.deleteAuthCode(event);
                } else if(object instanceof AuthRefTokenCreatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthRefTokenCreatedEvent = {}", event);
                    // dbProvider.createRefToken(event);
                } else if(object instanceof AuthRefTokenDeletedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthRefTokenDeletedEvent = {}", event);
                    // dbProvider.deleteRefToken(event);
                } else if(object instanceof AuthProviderCreatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthProviderCreatedEvent = {}", event);
                    dbProvider.createAuthProvider(event);
                } else if(object instanceof AuthProviderRotatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthProviderRotatedEvent = {}", event);
                    dbProvider.rotateAuthProvider(event);
                } else if(object instanceof AuthProviderUpdatedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthProviderUpdatedEvent = {}", event);
                    dbProvider.updateAuthProvider(event);
                } else if(object instanceof AuthProviderDeletedEvent event) {
                    if (logger.isTraceEnabled()) logger.trace("AuthProviderDeletedEvent = {}", event);
                    dbProvider.deleteAuthProvider(event);
                } else if(object instanceof ProductCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ProductCreatedEvent = {}", event);
                    dbProvider.createProduct(event);
                } else if(object instanceof ProductUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ProductUpdatedEvent = {}", event);
                    dbProvider.updateProduct(event);
                } else if(object instanceof ProductDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ProductDeletedEvent = {}", event);
                    dbProvider.deleteProduct(event);
                } else if(object instanceof PipelineCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PipelineCreatedEvent = {}", event);
                    dbProvider.createPipeline(event);
                } else if(object instanceof PipelineUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PipelineUpdatedEvent = {}", event);
                    dbProvider.updatePipeline(event);
                } else if(object instanceof PipelineDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PipelineDeletedEvent = {}", event);
                    dbProvider.deletePipeline(event);
                } else if(object instanceof PlatformCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PlatformCreatedEvent = {}", event);
                    dbProvider.createPlatform(event);
                } else if(object instanceof PlatformUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PlatformUpdatedEvent = {}", event);
                    dbProvider.updatePlatform(event);
                } else if(object instanceof PlatformDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("PlatformDeletedEvent = {}", event);
                    dbProvider.deletePlatform(event);
                } else if(object instanceof InstanceCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("InstanceCreatedEvent = {}", event);
                    dbProvider.createInstance(event);
                } else if(object instanceof InstanceUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("InstanceUpdatedEvent = {}", event);
                    dbProvider.updateInstance(event);
                } else if(object instanceof InstanceDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("InstanceDeletedEvent = {}", event);
                    dbProvider.deleteInstance(event);
                } else if(object instanceof InstanceApiCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("InstanceApiCreatedEvent = {}", event);
                    dbProvider.createInstanceApi(event);
                } else if(object instanceof InstanceApiUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("InstanceApiUpdatedEvent = {}", event);
                    dbProvider.updateInstanceApi(event);
                } else if(object instanceof InstanceApiDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("InstanceApiDeletedEvent = {}", event);
                    dbProvider.deleteInstanceApi(event);
                } else if(object instanceof InstanceAppCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("InstanceAppCreatedEvent = {}", event);
                    dbProvider.createInstanceApp(event);
                } else if(object instanceof InstanceAppUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("InstanceAppUpdatedEvent = {}", event);
                    dbProvider.updateInstanceApp(event);
                } else if(object instanceof InstanceAppDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("InstanceAppDeletedEvent = {}", event);
                    dbProvider.deleteInstanceApp(event);
                } else if(object instanceof DeploymentCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("DeploymentCreatedEvent = {}", event);
                    dbProvider.createDeployment(event);
                } else if(object instanceof DeploymentUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("DeploymentUpdatedEvent = {}", event);
                    dbProvider.updateDeployment(event);
                } else if(object instanceof DeploymentDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("DeploymentDeletedEvent = {}", event);
                    dbProvider.deleteDeployment(event);
                } else if(object instanceof ConfigCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigCreatedEvent = {}", event);
                    dbProvider.createConfig(event);
                } else if(object instanceof ConfigUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigUpdatedEvent = {}", event);
                    dbProvider.updateConfig(event);
                } else if(object instanceof ConfigDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigDeletedEvent = {}", event);
                    dbProvider.deleteConfig(event);
                } else if(object instanceof ConfigPropertyCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigPropertyCreatedEvent = {}", event);
                    dbProvider.createConfigProperty(event);
                } else if(object instanceof ConfigPropertyUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigPropertyUpdatedEvent = {}", event);
                    dbProvider.updateConfigProperty(event);
                } else if(object instanceof ConfigPropertyDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigPropertyDeletedEvent = {}", event);
                    dbProvider.deleteConfigProperty(event);
                } else if(object instanceof ConfigEnvironmentCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigEnvironmentCreatedEvent = {}", event);
                    dbProvider.createConfigEnvironment(event);
                } else if(object instanceof ConfigEnvironmentUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigEnvironmentUpdatedEvent = {}", event);
                    dbProvider.updateConfigEnvironment(event);
                } else if(object instanceof ConfigEnvironmentDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigEnvironmentDeletedEvent = {}", event);
                    dbProvider.deleteConfigEnvironment(event);
                } else if(object instanceof ConfigInstanceApiCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigInstanceApiCreatedEvent = {}", event);
                    dbProvider.createConfigInstanceApi(event);
                } else if(object instanceof ConfigInstanceApiUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigInstanceApiUpdatedEvent = {}", event);
                    dbProvider.updateConfigInstanceApi(event);
                } else if(object instanceof ConfigInstanceApiDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigInstanceApiDeletedEvent = {}", event);
                    dbProvider.deleteConfigInstanceApi(event);
                } else if(object instanceof ConfigInstanceAppCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigInstanceAppCreatedEvent = {}", event);
                    dbProvider.createConfigInstanceApp(event);
                } else if(object instanceof ConfigInstanceAppUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigInstanceAppUpdatedEvent = {}", event);
                    dbProvider.updateConfigInstanceApp(event);
                } else if(object instanceof ConfigInstanceAppDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigInstanceAppDeletedEvent = {}", event);
                    dbProvider.deleteConfigInstanceApp(event);
                } else if(object instanceof ConfigInstanceCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigInstanceCreatedEvent = {}", event);
                    dbProvider.createConfigInstance(event);
                } else if(object instanceof ConfigInstanceUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigInstanceUpdatedEvent = {}", event);
                    dbProvider.updateConfigInstance(event);
                } else if(object instanceof ConfigInstanceDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigInstanceDeletedEvent = {}", event);
                    dbProvider.deleteConfigInstance(event);
                } else if(object instanceof ConfigProductCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigProductCreatedEvent = {}", event);
                    dbProvider.createConfigProduct(event);
                } else if(object instanceof ConfigProductUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigProductUpdatedEvent = {}", event);
                    dbProvider.updateConfigProduct(event);
                } else if(object instanceof ConfigProductDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigProductDeletedEvent = {}", event);
                    dbProvider.deleteConfigProduct(event);
                } else if(object instanceof ConfigProductVersionCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigProductVersionCreatedEvent = {}", event);
                    dbProvider.createConfigProductVersion(event);
                } else if(object instanceof ConfigProductVersionUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigProductVersionUpdatedEvent = {}", event);
                    dbProvider.updateConfigProductVersion(event);
                } else if(object instanceof ConfigProductVersionDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ConfigProductVersionDeletedEvent = {}", event);
                    dbProvider.deleteConfigProductVersion(event);
                } else if(object instanceof AppCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AppCreatedEvent = {}", event);
                    dbProvider.createApp(event);
                } else if(object instanceof AppUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AppUpdatedEvent = {}", event);
                    dbProvider.updateApp(event);
                } else if(object instanceof AppDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("AppDeletedEvent = {}", event);
                    dbProvider.deleteApp(event);
                } else if(object instanceof ClientCreatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ClientCreatedEvent = {}", event);
                    dbProvider.createClient(event);
                } else if(object instanceof ClientUpdatedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ClientUpdatedEvent = {}", event);
                    dbProvider.updateClient(event);
                } else if(object instanceof ClientDeletedEvent event) {
                    if(logger.isTraceEnabled()) logger.trace("ClientDeletedEvent = {}", event);
                    dbProvider.deleteClient(event);
                }
            } catch (Exception e) {
                logger.error("Exception:", e);
            }

            */
        }

        @Override
        public void close() {
            if(logger.isInfoEnabled()) logger.info("Closing processor...");
        }
    }
    @Override
    public void start(String ip, int port) {
        if(logger.isDebugEnabled()) logger.debug("UserStreams is starting...");
        startUserStreams(ip, port);
    }

    @Override
    public void close() {
        if(logger.isDebugEnabled()) logger.debug("UserStreams is closing...");
        userStreams.close();
    }

    private static List<Map<String, Object>> parseSpec(String hostId, String apiId, String apiVersion, String apiType, String spec) {
        List<Map<String, Object>> endpoints = new ArrayList<>();
        switch (apiType) {
            case "openapi":
                OpenApiHelper helper = new OpenApiHelper(spec);
                String basePath = helper.basePath;
                Map<String, Path> paths = helper.openApi3.getPaths();
                for (Map.Entry<String, Path> pathPair : paths.entrySet()) {
                    String path = pathPair.getKey();
                    for (Map.Entry<String, Operation> entries : pathPair.getValue().getOperations().entrySet()) {
                        Map<String, Object> endpointMap = new HashMap<>();
                        String method = entries.getKey();
                        Operation operation = entries.getValue();
                        String endpointName = operation.getOperationId();
                        String endpointDesc = operation.getSummary();
                        String endpoint = basePath + path + "@" + method;
                        Collection<SecurityRequirement> securityRequirements = operation.getSecurityRequirements();
                        List<String> scopes = null;
                        if(securityRequirements != null) {
                            for(SecurityRequirement requirement: securityRequirements) {
                                SecurityParameter securityParameter = null;
                                for(String oauth2Name: helper.oauth2Names) {
                                    securityParameter = requirement.getRequirement(oauth2Name);
                                    if(securityParameter != null) break;
                                }
                                if(securityParameter != null) scopes = securityParameter.getParameters();
                                if(scopes != null) break;
                            }
                        }
                        endpointMap.put("scopes", scopes);
                        endpointMap.put("endpointId", UuidUtil.getUUID().toString());
                        endpointMap.put("endpoint", endpoint);
                        endpointMap.put("httpMethod", method);
                        endpointMap.put("endpointPath", path);
                        endpointMap.put("endpointName", endpointName);
                        endpointMap.put("endpointDesc", endpointDesc);
                        endpoints.add(endpointMap);
                    }
                }
                break;
            case "graphql":

                break;
            case "hybrid":
                Map<String, Object> specMap = PortalUtil.yamlToMap(spec);
                String host = (String)specMap.get("host");
                String service = (String)specMap.get("service");
                List<Map<String, Object>> actionList = (List)specMap.get("action");
                for(Map<String, Object> action: actionList) {
                    Map<String, Object> endpointMap = new HashMap<>();
                    String endpoint = host + "/" + service + "/" + action.get("name") + "/" + action.get("version");
                    endpointMap.put("endpointId", UuidUtil.getUUID().toString());
                    endpointMap.put("endpoint", endpoint);
                    endpointMap.put("endpointName", action.get("handler"));
                    if(action.get("scope") != null) {
                        List<String> list = new ArrayList<String>();
                        list.add((String)action.get("scope"));
                        endpointMap.put("scopes", list);
                    }
                    endpoints.add(endpointMap);
                }
                break;
            default:
                logger.error("ServiceSpecUpdatedEvent with unknown apiType {} for hostId {} apiId {} apiVersion {}", apiType, hostId, apiId, apiVersion);
        }
        return endpoints;
    }
}
