package net.lightapi.portal.user.query;

import com.networknt.config.Config;
import com.networknt.kafka.common.KafkaStreamsConfig;
import com.networknt.kafka.streams.LightStreams;
import com.networknt.utility.ByteUtil;
import net.lightapi.portal.PortalConfig;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.common.serialization.Serdes;
import org.apache.kafka.streams.*;
import org.apache.kafka.streams.errors.StreamsUncaughtExceptionHandler;
import org.apache.kafka.streams.processor.*;
import org.apache.kafka.streams.state.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.Properties;

public class NonceQueryStreams implements LightStreams {
    static private final Logger logger = LoggerFactory.getLogger(NonceQueryStreams.class);
    static final KafkaStreamsConfig streamsConfig = (KafkaStreamsConfig) Config.getInstance().getJsonObjectConfig(KafkaStreamsConfig.CONFIG_NAME, KafkaStreamsConfig.class);
    static final PortalConfig portalConfig = (PortalConfig) Config.getInstance().getJsonObjectConfig(PortalConfig.CONFIG_NAME, PortalConfig.class);

    static private final String nonce = "user-nonce-store";

    KafkaStreams nonceStreams;

    public NonceQueryStreams() {
        logger.info("NonceQueryStreams is created");
    }

    public ReadOnlyKeyValueStore<String, Long> getUserNonceStore() {
        QueryableStoreType<ReadOnlyKeyValueStore<String, Long>> queryableStoreType = QueryableStoreTypes.keyValueStore();
        StoreQueryParameters<ReadOnlyKeyValueStore<String, Long>> sqp = StoreQueryParameters.fromNameAndType(nonce, queryableStoreType);
        return nonceStreams.store(sqp);
    }

    public KeyQueryMetadata getUserNonceStreamsMetadata(String email) {
        return nonceStreams.queryMetadataForKey(nonce, email, Serdes.String().serializer());
    }

    private void startNonceStreams(String ip, int port) {

        final StreamsBuilder builder = new StreamsBuilder();

        StoreBuilder<KeyValueStore<String, Long>> keyValueNonceStoreBuilder =
                Stores.keyValueStoreBuilder(Stores.persistentKeyValueStore(nonce),
                        Serdes.String(),
                        Serdes.Long());
        builder.addStateStore(keyValueNonceStoreBuilder);

        builder.stream(portalConfig.getNonceTopic()).process(new ProcessorSupplier() {
            @Override
            public Processor get() {
                return new NonceQueryStreams.NonceEventProcessor();
            }
        }, nonce);

        final Topology topology = builder.build();

        Properties streamsProps = new Properties();
        streamsProps.putAll(streamsConfig.getProperties());
        streamsProps.put(StreamsConfig.DEFAULT_KEY_SERDE_CLASS_CONFIG, Serdes.ByteArray().getClass());
        streamsProps.put(StreamsConfig.DEFAULT_VALUE_SERDE_CLASS_CONFIG, Serdes.ByteArray().getClass());
        streamsProps.put(StreamsConfig.APPLICATION_ID_CONFIG, portalConfig.getNonceApplicationId());
        streamsProps.put(StreamsConfig.APPLICATION_SERVER_CONFIG, ip + ":" + port);
        nonceStreams = new KafkaStreams(topology, streamsProps);
        nonceStreams.setUncaughtExceptionHandler(ex -> {
            logger.error("Kafka-Streams uncaught exception occurred. Stream will be replaced with new thread", ex);
            return StreamsUncaughtExceptionHandler.StreamThreadExceptionResponse.REPLACE_THREAD;
        });

        if(streamsConfig.isCleanUp()) {
            nonceStreams.cleanUp();
        }
        nonceStreams.start();
    }

    public static class NonceEventProcessor extends AbstractProcessor<byte[], byte[]> {

        private ProcessorContext pc;
        private KeyValueStore<String, Long> nonceStore;

        public NonceEventProcessor() {
        }

        @Override
        public void init(ProcessorContext pc) {

            this.pc = pc;
            this.nonceStore = (KeyValueStore<String, Long>) pc.getStateStore(nonce);
            if(logger.isInfoEnabled()) logger.info("Processor initialized");
        }

        @Override
        public void process(byte[] key, byte[] value) {
            nonceStore.put(new String(key), ByteUtil.bytesToLong(value));
        }

        @Override
        public void close() {
            if(logger.isInfoEnabled()) logger.info("Closing processor...");
        }
    }
    @Override
    public void start(String ip, int port) {
        if(logger.isDebugEnabled()) logger.debug("NonceStreams is starting...");
        startNonceStreams(ip, port);
    }

    @Override
    public void close() {
        if(logger.isDebugEnabled()) logger.debug("NonceStreams is closing...");
        nonceStreams.close();
    }


}
