
package net.lightapi.portal.service.query.handler;

import com.networknt.config.JsonMapper;
import com.networknt.db.provider.DbProvider;
import com.networknt.monad.Result;
import com.networknt.service.SingletonServiceFactory;
import com.networknt.utility.NioUtils;
import com.networknt.rpc.HybridHandler;
import com.networknt.rpc.router.ServiceHandler;
import java.nio.ByteBuffer;
import java.util.*;
import java.util.stream.Collectors;

import io.undertow.server.HttpServerExchange;
import net.lightapi.portal.db.PortalDbProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * return the permission of the service per api endpoint. It is used during the service startup to load permission
 * for fine-grained access control.
*/
@ServiceHandler(id="lightapi.net/service/getServicePermission/0.1.0")
public class GetServicePermission implements HybridHandler {
    private static final Logger logger = LoggerFactory.getLogger(GetServicePermission.class);
    public static final String OBJECT_NOT_FOUND = "ERR11637";
    public static PortalDbProvider dbProvider = (PortalDbProvider) SingletonServiceFactory.getBean(DbProvider.class);

    @Override
    public ByteBuffer handle(HttpServerExchange exchange, Object input) {
        if (logger.isTraceEnabled()) logger.trace("input = {}", input);
        Map<String, Object> map = (Map<String, Object>) input;
        String hostId = (String) map.get("hostId");
        String apiId = (String) map.get("apiId");
        String apiVersion = (String) map.get("apiVersion");
        if (logger.isTraceEnabled())
            logger.trace("hostId = {} apiId = {} apiVersion = {}", hostId, apiId, apiVersion);
        Result<String> result = dbProvider.queryServicePermission(hostId, apiId, apiVersion);
        if (result.isFailure()) {
            return NioUtils.toByteBuffer(getStatus(exchange, OBJECT_NOT_FOUND, "service permission", "hostId: " + hostId + " apiId: " + apiId + " apiVersion: " + apiVersion));
        } else {
            String servicePermission = result.getResult();
            if (logger.isTraceEnabled()) logger.trace("servicePermission = {}", servicePermission);
            // transform the permission to concat roleIds, groupIds, positionIds, userIds and attributeId/Value pairs.
            List<Map<String, Object>> output = transformJsonList(JsonMapper.string2List(servicePermission));
            // enrich with the filter information for the endpoints.
            Result<List<String>> filterResult = dbProvider.queryServiceFilter(hostId, apiId, apiVersion);
            if (filterResult.isFailure()) {
                return NioUtils.toByteBuffer(getStatus(exchange, OBJECT_NOT_FOUND, "service filter", "hostId: " + hostId + " apiId: " + apiId + " apiVersion: " + apiVersion));
            }
            List<String> filters = filterResult.getResult();
            if (logger.isTraceEnabled()) logger.trace("filters = {}", filters);
            List<Map<String, Object>> enrichedOutput = enrichWithFilters(output, filters);
            String outputJson = JsonMapper.toJson(enrichedOutput);
            if (logger.isTraceEnabled()) logger.trace("outputJson = {}", outputJson);
            return NioUtils.toByteBuffer(outputJson);
        }
    }

    public static List<Map<String, Object>> transformJsonList(List<Map<String, Object>> inputList) {
        return inputList.stream()
                .map(GetServicePermission::transformMap)
                .filter(transformedMap -> !transformedMap.isEmpty()) // Filter empty maps
                .collect(Collectors.toList());
    }

    private static Map<String, Object> transformMap(Map<String, Object> inputMap) {
        Map<String, Object> transformedMap = new HashMap<>(inputMap);
        for (String key : List.of("roles", "positions", "groups", "users")) {
            if (transformedMap.containsKey(key)) {
                String transformedValue = transformListMapToString(transformedMap.get(key));
                if (!transformedValue.isEmpty()) {
                    transformedMap.put(key, transformedValue);
                } else {
                    transformedMap.remove(key);
                }

            }
        }

        if (transformedMap.containsKey("attributes")) {
            List<Map<String, Object>> transformedAttributes = transformAttributes((List<Map<String, String>>) transformedMap.get("attributes"));
            if (!transformedAttributes.isEmpty()) {
                transformedMap.put("attributes", transformedAttributes);
            } else {
                transformedMap.remove("attributes");
            }
        }
        return transformedMap;

    }

    private static String transformListMapToString(Object listOrString) {
        if (listOrString instanceof String) {
            return (String) listOrString;
        } else if (listOrString instanceof List) {
            List<Map<String, Object>> listOfMaps = (List<Map<String, Object>>) listOrString;
            return listOfMaps.stream()
                    .filter(Objects::nonNull)
                    .map(map -> map.values().stream().filter(Objects::nonNull).findFirst().orElse(""))
                    .filter(value -> !value.toString().isEmpty())
                    .map(Object::toString)
                    .collect(Collectors.joining(" "));

        }
        return "";
    }


    private static List<Map<String, Object>> transformAttributes(List<Map<String, String>> attributesList) {
        List<Map<String, Object>> attributes = new ArrayList<>();
        if (attributesList != null) {
            for (Map<String, String> attribute : attributesList) {
                Map<String, Object> attributeMap = new HashMap<>();
                if (attribute != null && attribute.get("attribute_id") != null && attribute.get("attribute_value") != null) {
                    attributeMap.put(attribute.get("attribute_id"), attribute.get("attribute_value"));
                }
                attributes.add(attributeMap);
            }
        }
        return attributes;
    }

    private static List<Map<String, Object>> enrichWithFilters(List<Map<String, Object>> output, List<String> filters) {
        // first transform the filters to a map for easy lookup.
        Map<String, Object> filtersMap = transformFilter(filters); // endpoint is the key and two filters are the value. (row and col)
        for (Map<String, Object> endpointMap : output) {
            String endpoint = (String) endpointMap.get("endpoint");
            Map<String, Object> map = (Map<String, Object>) filtersMap.get(endpoint);
            if(map != null) endpointMap.putAll(map);
        }
        return output;
    }

    public static Map<String, Object> transformFilter(List<String> input) {
        Map<String, Object> result = new HashMap<>();
        Map<String, Object> endpointFilters = new HashMap<>();
        Map<String, Object> rowMap = new HashMap<>();
        Map<String, Object> colMap = new HashMap<>();
        endpointFilters.put("row", rowMap);
        endpointFilters.put("col", colMap);

        for (String jsonStr : input) {
            try {
                Map<String, Object> parsedMap = JsonMapper.string2Map(jsonStr);
                parsedMap.forEach((type, typeData) -> {
                    if (typeData instanceof List) {
                        List<Map<String, Object>> dataList = (List<Map<String, Object>>) typeData;
                        String endpoint = null;
                        for (Map<String, Object> item : dataList) {
                            endpoint = (String) item.get("endpoint");
                            if (type.equals("role_row")) {
                                processRoleRow(rowMap, item);
                            } else if (type.equals("role_col")) {
                                processRoleCol(colMap, item);
                            }
                        }
                        if (endpoint != null) {
                            result.put(endpoint, endpointFilters);
                        }
                    }
                });
            } catch (Exception e) {
                // Handle JSON parsing exception
                e.printStackTrace();
            }
        }
        return result;
    }

    private static void processRoleRow(Map<String, Object> rowMap, Map<String, Object> item) {
        String roleId = (String) item.get("roleId");

        // Get the "role" map (cast to Map) and then use computeIfAbsent on it
        Map<String, Object> roleMap = (Map<String, Object>) rowMap.computeIfAbsent("role", k -> new HashMap<>());

        // Get the list of filters for the roleId, creating a new list if it doesn't exist
        List<Map<String, String>> roleFilters = (List<Map<String, String>>) roleMap.computeIfAbsent(roleId, k -> new ArrayList<>());

        Map<String, String> filter = new HashMap<>();
        filter.put("colName", (String) item.get("colName"));
        filter.put("operator", (String) item.get("operator"));
        filter.put("colValue", (String) item.get("colValue"));

        // Add the filter to the list of filters
        roleFilters.add(filter);
    }

    private static void processRoleCol(Map<String, Object> colMap, Map<String, Object> item) {
        String roleId = (String) item.get("roleId");
        // Get the "role" map (cast to Map) and then use put directly
        Map<String, Object> roleMap = (Map<String, Object>) colMap.computeIfAbsent("role", k -> new HashMap<>());
        roleMap.put(roleId, item.get("columns"));
    }}
