
package net.lightapi.portal.service.command.handler;

import com.networknt.monad.Failure;
import com.networknt.monad.Result;
import com.networknt.monad.Success;
import com.networknt.rpc.router.ServiceHandler;
import com.networknt.status.Status;
import io.undertow.server.HttpServerExchange;
import net.lightapi.portal.PortalConstants;
import net.lightapi.portal.PortalUtil;
import net.lightapi.portal.command.AbstractCommandHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;

/**
 * update service version
 */
@ServiceHandler(id="lightapi.net/service/updateServiceVersion/0.1.0")
public class UpdateServiceVersion extends AbstractCommandHandler {
    private static final Logger logger = LoggerFactory.getLogger(UpdateServiceVersion.class);
    private static final String INVALID_CONTENT_FORMAT = "ERR11641";
    private static final String INVALID_REMOTE_URL = "ERR11642";

    @Override
    protected String getCloudEventType() {
        return PortalConstants.SERVICE_VERSION_UPDATED_EVENT;
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected Result<Map<String, Object>> enrichInput(HttpServerExchange exchange, Map<String, Object> map) {
        // we need to convert the spec url to spec content before sending the event to the kafka topic.
        String spec = (String)map.get("spec");
        if(spec == null || spec.isBlank()) {
            // try to get the specLink and download the spec content.
            String specLink = (String)map.remove("specLink");
            if(specLink != null && !specLink.isBlank()) {
                specLink = specLink.trim();
                if(specLink.endsWith(".yml") || specLink.endsWith(".yaml")) {
                    try {
                        spec = PortalUtil.readUrl(specLink);
                        // check the spec content to make sure it is a valid yaml content.
                        if(PortalUtil.isValidYaml(spec)) {
                            map.put("spec", spec);
                        } else {
                            return Failure.of(new Status(INVALID_CONTENT_FORMAT, "YAML"));
                        }
                    } catch (Exception e) {
                        return Failure.of(new Status(INVALID_REMOTE_URL, specLink));
                    }
                } else {
                    return Failure.of(new Status(INVALID_CONTENT_FORMAT, "YAML"));
                }
            }
        } else {
            // check the spec content to make sure it is a valid yaml content.
            if(!PortalUtil.isValidYaml(spec)) {
                return Failure.of(new Status(INVALID_CONTENT_FORMAT, "YAML"));
            }
        }
        return Success.of(map);
    }
}
