
package net.lightapi.portal.oauth.query.handler;

import com.networknt.config.JsonMapper;
import com.networknt.db.provider.DbProvider;
import com.networknt.httpstring.AttachmentConstants;
import com.networknt.monad.Result;
import com.networknt.service.SingletonServiceFactory;
import com.networknt.utility.NioUtils;
import com.networknt.rpc.HybridHandler;
import com.networknt.rpc.router.ServiceHandler;
import java.nio.ByteBuffer;
import java.util.Map;

import io.undertow.server.HttpServerExchange;
import net.lightapi.portal.db.PortalDbProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Get registered client detail by clientId. There are two consumers for this API. The light-portal UI
 * and light-oauth2 for client authentication. The first case will have an authorization code token with
 * a userId and the second is client credentials token without userId. Based on it, we will return the
 * client secret to the light-oauth2 only.
 *
 * @author Steve Hu
 */
@ServiceHandler(id="lightapi.net/oauth/getClientById/0.1.0")
public class GetClientById implements HybridHandler {
    private static final Logger logger = LoggerFactory.getLogger(GetClientById.class);
    private static final String OBJECT_NOT_FOUND = "ERR11637";
    private static final String PERMISSION_DENIED = "ERR11620";
    public static PortalDbProvider dbProvider = (PortalDbProvider) SingletonServiceFactory.getBean(DbProvider.class);

    @Override
    public ByteBuffer handle(HttpServerExchange exchange, Object input)  {
        if(logger.isTraceEnabled()) logger.trace("input = " + input);
        Map<String, Object> map = (Map<String, Object>)input;
        String clientId = (String)map.get("clientId");
        Result<String> result = dbProvider.queryClientByClientId(clientId);
        if(result.isSuccess()) {
            Map<String, Object> auditInfo = exchange.getAttachment(AttachmentConstants.AUDIT_INFO);
            Map<String, Object> clientMap = JsonMapper.string2Map(result.getResult());
            if(auditInfo != null) {
                String userId = (String)auditInfo.get("user_id");
                if(userId != null) {
                    // remove the clientSecret from the data for security.
                    clientMap.remove("clientSecret");
                    return NioUtils.toByteBuffer(JsonMapper.toJson(clientMap));
                } else {
                    // return the entire client to the light-oauth2
                    return NioUtils.toByteBuffer(JsonMapper.toJson(clientMap));
                }
            } else {
                return NioUtils.toByteBuffer(getStatus(exchange, PERMISSION_DENIED, "Missing token"));
            }
        } else {
            return NioUtils.toByteBuffer(getStatus(exchange, OBJECT_NOT_FOUND, "application with clientId ", clientId));
        }
    }
}
