/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.common.archive;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.zip.DataFormatException;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;
import org.wildfly.common.Assert;
import org.wildfly.common.archive.ByteBufferInputStream;
import org.wildfly.common.archive.HugeIndex;
import org.wildfly.common.archive.Index;
import org.wildfly.common.archive.JDKSpecific;
import org.wildfly.common.archive.LargeIndex;
import org.wildfly.common.archive.TinyIndex;

public final class Archive
implements Closeable {
    public static final int GP_ENCRYPTED = 1;
    public static final int GP_IMPLODE_8K_DICTIONARY = 2;
    public static final int GP_IMPLODE_3_TREES = 4;
    public static final int GP_DEFLATE_COMP_OPT_MASK = 6;
    public static final int GP_DEFLATE_COMP_OPT_NORMAL = 0;
    public static final int GP_DEFLATE_COMP_OPT_MAXIMUM = 2;
    public static final int GP_DEFLATE_COMP_OPT_FAST = 4;
    public static final int GP_DEFLATE_COMP_OPT_SUPER_FAST = 6;
    public static final int GP_LZMA_EOS_USED = 2;
    public static final int GP_LATE_SIZES = 8;
    public static final int GP_COMPRESSED_PATCHED = 32;
    public static final int GP_STRONG_ENCRYPTION = 64;
    public static final int GP_UTF_8 = 2048;
    public static final int GP_CD_MASKED = 8192;
    public static final int METHOD_STORED = 0;
    public static final int METHOD_SHRINK = 1;
    public static final int METHOD_REDUCE_1 = 2;
    public static final int METHOD_REDUCE_2 = 3;
    public static final int METHOD_REDUCE_3 = 4;
    public static final int METHOD_REDUCE_4 = 5;
    public static final int METHOD_IMPLODE = 6;
    public static final int METHOD_DEFLATE = 8;
    public static final int METHOD_DEFLATE64 = 9;
    public static final int METHOD_BZIP2 = 12;
    public static final int METHOD_LZMA = 14;
    public static final int MADE_BY_MS_DOS = 0;
    public static final int MADE_BY_UNIX = 3;
    public static final int MADE_BY_NTFS = 10;
    public static final int MADE_BY_OS_X = 19;
    public static final int SIG_LH = 67324752;
    public static final int LH_SIGNATURE = 0;
    public static final int LH_MIN_VERSION = 4;
    public static final int LH_GP_BITS = 6;
    public static final int LH_COMP_METHOD = 8;
    public static final int LH_MOD_TIME = 10;
    public static final int LH_MOD_DATE = 12;
    public static final int LH_CRC_32 = 14;
    public static final int LH_COMPRESSED_SIZE = 18;
    public static final int LH_UNCOMPRESSED_SIZE = 22;
    public static final int LH_FILE_NAME_LENGTH = 26;
    public static final int LH_EXTRA_LENGTH = 28;
    public static final int LH_END = 30;
    public static final int SIG_DD = 134695760;
    public static final int DD_SIGNATURE = 0;
    public static final int DD_CRC_32 = 4;
    public static final int DD_COMPRESSED_SIZE = 8;
    public static final int DD_UNCOMPRESSED_SIZE = 12;
    public static final int DD_END = 16;
    public static final int DD_ZIP64_COMPRESSED_SIZE = 8;
    public static final int DD_ZIP64_UNCOMPRESSED_SIZE = 16;
    public static final int DD_ZIP64_END = 24;
    public static final int SIG_CDE = 33639248;
    public static final int CDE_SIGNATURE = 0;
    public static final int CDE_VERSION_MADE_BY = 4;
    public static final int CDE_VERSION_NEEDED = 6;
    public static final int CDE_GP_BITS = 8;
    public static final int CDE_COMP_METHOD = 10;
    public static final int CDE_MOD_TIME = 12;
    public static final int CDE_MOD_DATE = 14;
    public static final int CDE_CRC_32 = 16;
    public static final int CDE_COMPRESSED_SIZE = 20;
    public static final int CDE_UNCOMPRESSED_SIZE = 24;
    public static final int CDE_FILE_NAME_LENGTH = 28;
    public static final int CDE_EXTRA_LENGTH = 30;
    public static final int CDE_COMMENT_LENGTH = 32;
    public static final int CDE_FIRST_DISK_NUMBER = 34;
    public static final int CDE_INTERNAL_ATTRIBUTES = 36;
    public static final int CDE_EXTERNAL_ATTRIBUTES = 38;
    public static final int CDE_LOCAL_HEADER_OFFSET = 42;
    public static final int CDE_END = 46;
    public static final int SIG_EOCD = 101010256;
    public static final int EOCD_SIGNATURE = 0;
    public static final int EOCD_DISK_NUMBER = 4;
    public static final int EOCD_CD_FIRST_DISK_NUMBER = 6;
    public static final int EOCD_CDE_COUNT_THIS_DISK = 8;
    public static final int EOCD_CDE_COUNT_ALL = 10;
    public static final int EOCD_CD_SIZE = 12;
    public static final int EOCD_CD_START_OFFSET = 16;
    public static final int EOCD_COMMENT_LENGTH = 20;
    public static final int EOCD_END = 22;
    public static final int EXT_ID_ZIP64 = 1;
    public static final int ZIP64_UNCOMPRESSED_SIZE = 0;
    public static final int ZIP64_COMPRESSED_SIZE = 8;
    public static final int ZIP64_LOCAL_HEADER_OFFSET = 16;
    public static final int ZIP64_FIRST_DISK_NUMBER = 24;
    public static final int ZIP64_END = 28;
    public static final int EXT_ID_UNIX = 13;
    public static final int UNIX_ACCESS_TIME = 0;
    public static final int UNIX_MODIFIED_TIME = 4;
    public static final int UNIX_UID = 8;
    public static final int UNIX_GID = 10;
    public static final int UNIX_END = 12;
    public static final int UNIX_DEV_MAJOR = 12;
    public static final int UNIX_DEV_MINOR = 16;
    public static final int UNIX_DEV_END = 20;
    public static final int SIG_EOCD_ZIP64 = 101075792;
    public static final int EOCD_ZIP64_SIGNATURE = 0;
    public static final int EOCD_ZIP64_SIZE = 4;
    public static final int EOCD_ZIP64_VERSION_MADE_BY = 12;
    public static final int EOCD_ZIP64_VERSION_NEEDED = 14;
    public static final int EOCD_ZIP64_DISK_NUMBER = 16;
    public static final int EOCD_ZIP64_CD_FIRST_DISK_NUMBER = 20;
    public static final int EOCD_ZIP64_CDE_COUNT_THIS_DISK = 24;
    public static final int EOCD_ZIP64_CDE_COUNT_ALL = 28;
    public static final int EOCD_ZIP64_CD_SIZE = 36;
    public static final int EOCD_ZIP64_CD_START_OFFSET = 44;
    public static final int EOCD_ZIP64_END = 52;
    public static final int SIG_EOCDL_ZIP64 = 117853008;
    public static final int EOCDL_ZIP64_SIGNATURE = 0;
    public static final int EOCDL_ZIP64_EOCD_DISK_NUMBER = 4;
    public static final int EOCDL_ZIP64_EOCD_OFFSET = 8;
    public static final int EOCDL_ZIP64_DISK_COUNT = 16;
    public static final int EOCDL_ZIP64_END = 20;
    private static final int BUF_SIZE_MAX = 0x40000000;
    private static final int BUF_SHIFT = Integer.numberOfTrailingZeros(0x40000000);
    private static final int BUF_SIZE_MASK = 0x3FFFFFFF;
    private final ByteBuffer[] bufs;
    private final long offset;
    private final long length;
    private final long cd;
    private final Index index;
    private static final ByteBuffer EMPTY_BUF = ByteBuffer.allocateDirect(0);

    private Archive(ByteBuffer[] bufs, long offset, long length, long cd2, Index index) {
        this.bufs = bufs;
        this.offset = offset;
        this.length = length;
        this.cd = cd2;
        this.index = index;
    }

    public static Archive open(Path path) throws IOException {
        try (FileChannel fc = FileChannel.open(path, StandardOpenOption.READ);){
            long size = fc.size();
            int bufCnt = Math.toIntExact(size + 0x3FFFFFFFL >> BUF_SHIFT);
            ByteBuffer[] array = new ByteBuffer[bufCnt];
            long offs = 0L;
            int idx = 0;
            while (size > 0x3FFFFFFFL) {
                array[idx++] = fc.map(FileChannel.MapMode.READ_ONLY, offs, 0x40000000L).order(ByteOrder.LITTLE_ENDIAN);
                size -= 0x40000000L;
                offs += 0x40000000L;
            }
            array[idx] = fc.map(FileChannel.MapMode.READ_ONLY, offs, size).order(ByteOrder.LITTLE_ENDIAN);
            Archive archive = Archive.open(array);
            return archive;
        }
    }

    public static Archive open(ByteBuffer buf) throws IOException {
        Assert.checkNotNullParam("buf", buf);
        if (buf.order() == ByteOrder.BIG_ENDIAN) {
            buf = buf.duplicate().order(ByteOrder.LITTLE_ENDIAN);
        }
        return Archive.open(new ByteBuffer[]{buf});
    }

    static Archive open(ByteBuffer[] bufs) throws IOException {
        return Archive.open(bufs, 0L, Archive.capacity(bufs));
    }

    static Archive open(ByteBuffer[] bufs, long offset, long length) throws IOException {
        long cd2;
        long eocd = length - 22L;
        while (Archive.getUnsignedInt(bufs, offset + eocd) != 101010256L) {
            if (eocd == 0L) {
                throw new IOException("Invalid archive");
            }
            --eocd;
        }
        int entries = Archive.getUnsignedShort(bufs, offset + eocd + 10L);
        if (Archive.getUnsignedShort(bufs, offset + eocd + 6L) != 0 || Archive.getUnsignedShort(bufs, offset + eocd + 4L) != 0 || entries != Archive.getUnsignedShort(bufs, offset + eocd + 8L)) {
            throw new IOException("Multi-disk archives are not supported");
        }
        long eocdLocZip64 = eocd - 20L;
        long eocdZip64 = -1L;
        if (Archive.getInt(bufs, offset + eocdLocZip64 + 0L) == 117853008) {
            if (Archive.getInt(bufs, offset + eocdLocZip64 + 16L) != 1 || Archive.getInt(bufs, offset + eocdLocZip64 + 4L) != 0) {
                throw new IOException("Multi-disk archives are not supported");
            }
            eocdZip64 = Archive.getLong(bufs, offset + eocdLocZip64 + 8L);
            if (Archive.getUnsignedInt(bufs, offset + eocdZip64 + 0L) != 101075792L) {
                eocdZip64 = -1L;
            }
        }
        if ((cd2 = Archive.getUnsignedInt(bufs, offset + eocd + 16L)) == 0xFFFFFFFFL && eocdZip64 != -1L) {
            cd2 = Archive.getLong(bufs, offset + eocdZip64 + 44L);
        }
        if (entries == 65535 && eocdZip64 != -1L) {
            long cnt = Archive.getUnsignedInt(bufs, offset + eocdZip64 + 28L);
            if (cnt > 0x7FFFFFFL) {
                throw new IOException("Archive has too many entries");
            }
            entries = (int)cnt;
        }
        Index index = length <= 65534L ? new TinyIndex(entries) : (length <= 0xFFFFFFFFEL ? new LargeIndex(entries) : new HugeIndex(entries));
        int mask = index.getMask();
        long cde = cd2;
        for (int i = 0; i < entries; ++i) {
            if (Archive.getInt(bufs, offset + cde + 0L) != 33639248) {
                throw new IOException("Archive appears to be corrupted");
            }
            int hc = Archive.getHashCodeOfEntry(bufs, offset + cde);
            index.put(hc & mask, cde);
            cde = cde + 46L + (long)Archive.getUnsignedShort(bufs, offset + cde + 28L) + (long)Archive.getUnsignedShort(bufs, offset + cde + 30L) + (long)Archive.getUnsignedShort(bufs, offset + cde + 32L);
        }
        return new Archive(bufs, offset, length, cd2, index);
    }

    private static String getNameOfEntry(ByteBuffer[] bufs, long cde) {
        boolean utf8;
        long name2 = cde + 46L;
        int nameLen = Archive.getUnsignedShort(bufs, cde + 28L);
        boolean bl = utf8 = (Archive.getUnsignedShort(bufs, cde + 8L) & 0x800) != 0;
        if (utf8) {
            return new String(Archive.getBytes(bufs, name2, nameLen), StandardCharsets.UTF_8);
        }
        char[] nameChars = new char[nameLen];
        for (int i = 0; i < nameLen; ++i) {
            nameChars[i] = Cp437.charFor(Archive.getUnsignedByte(bufs, name2 + (long)i));
        }
        return new String(nameChars);
    }

    private static int getHashCodeOfEntry(ByteBuffer[] bufs, long cde) {
        long name2 = cde + 46L;
        int nameLen = Archive.getUnsignedShort(bufs, cde + 28L);
        boolean utf8 = (Archive.getUnsignedShort(bufs, cde + 8L) & 0x800) != 0;
        int hc = 0;
        if (utf8) {
            for (int i = 0; i < nameLen; i += Utf8.getByteCount(Archive.getUnsignedByte(bufs, name2 + (long)i))) {
                int cp = Utf8.codePointAt(bufs, name2 + (long)i);
                if (Character.isSupplementaryCodePoint(cp)) {
                    hc = hc * 31 + Character.highSurrogate(cp);
                    hc = hc * 31 + Character.lowSurrogate(cp);
                    continue;
                }
                hc = hc * 31 + cp;
            }
        } else {
            for (int i = 0; i < nameLen; ++i) {
                hc = hc * 31 + Cp437.charFor(Archive.getUnsignedByte(bufs, name2 + (long)i));
            }
        }
        return hc;
    }

    public long getFirstEntryHandle() {
        return this.cd;
    }

    public long getNextEntryHandle(long entryHandle) {
        long next = entryHandle + 46L + (long)Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 28L) + (long)Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 30L);
        if (next >= this.length || Archive.getInt(this.bufs, this.offset + next + 0L) != 33639248) {
            return -1L;
        }
        return next;
    }

    public long getEntryHandle(String fileName) {
        int mask = this.index.getMask();
        int base2 = fileName.hashCode();
        for (int i = 0; i < mask; ++i) {
            long entryHandle = this.index.get(base2 + i & mask);
            if (entryHandle == -1L) {
                return -1L;
            }
            if (!this.entryNameEquals(entryHandle, fileName)) continue;
            return entryHandle;
        }
        return -1L;
    }

    public boolean entryNameEquals(long entryHandle, String fileName) {
        int i;
        long name2 = entryHandle + 46L;
        int nameLen = Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 28L);
        boolean utf8 = (Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 8L) & 0x800) != 0;
        int length = fileName.length();
        if (utf8) {
            long i2;
            int j = 0;
            for (i2 = 0L; i2 < (long)nameLen && j < length; i2 += (long)Utf8.getByteCount(Archive.getUnsignedByte(this.bufs, this.offset + name2 + i2))) {
                if (Utf8.codePointAt(this.bufs, this.offset + name2 + i2) != fileName.codePointAt(j)) {
                    return false;
                }
                j = fileName.offsetByCodePoints(j, 1);
            }
            return i2 == (long)nameLen && j == length;
        }
        int j = 0;
        for (i = 0; i < nameLen && j < length; ++i) {
            if (Cp437.charFor(Archive.getUnsignedByte(this.bufs, this.offset + (long)i + entryHandle + 46L)) != fileName.codePointAt(j)) {
                return false;
            }
            j = fileName.offsetByCodePoints(j, 1);
        }
        return i == nameLen && j == length;
    }

    private long getLocalHeader(long entryHandle) {
        long zip64;
        long lh = Archive.getUnsignedInt(this.bufs, this.offset + entryHandle + 42L);
        if (lh == 0xFFFFFFFFL && (zip64 = this.getExtraRecord(entryHandle, 1)) != -1L) {
            lh = Archive.getLong(this.bufs, this.offset + zip64 + 16L);
        }
        return lh;
    }

    public String getEntryName(long entryHandle) {
        return Archive.getNameOfEntry(this.bufs, entryHandle);
    }

    public ByteBuffer getEntryContents(long entryHandle) throws IOException {
        long size = this.getUncompressedSize(entryHandle);
        long compSize = this.getCompressedSize(entryHandle);
        if (size > 0x10000000L || compSize > 0x10000000L) {
            throw new IOException("Entry is too large to read into RAM");
        }
        long localHeader = this.getLocalHeader(entryHandle);
        if ((Archive.getUnsignedShort(this.bufs, this.offset + localHeader + 6L) & 0x41) != 0) {
            throw new IOException("Cannot read encrypted entries");
        }
        long offset = this.getDataOffset(localHeader);
        int method = this.getCompressionMethod(entryHandle);
        switch (method) {
            case 0: {
                return Archive.bufferOf(this.bufs, this.offset + offset, (int)size);
            }
            case 8: {
                Inflater inflater = new Inflater(true);
                try {
                    ByteBuffer byteBuffer = JDKSpecific.inflate(inflater, this.bufs, this.offset + offset, (int)compSize, (int)size);
                    return byteBuffer;
                }
                catch (DataFormatException e) {
                    throw new IOException(e);
                }
                finally {
                    inflater.end();
                }
            }
        }
        throw new IOException("Unsupported compression scheme");
    }

    private long getDataOffset(long localHeader) {
        return localHeader + 30L + (long)Archive.getUnsignedShort(this.bufs, this.offset + localHeader + 26L) + (long)Archive.getUnsignedShort(this.bufs, this.offset + localHeader + 28L);
    }

    public InputStream getEntryStream(long entryHandle) throws IOException {
        long size = this.getCompressedSize(entryHandle);
        long localHeader = this.getLocalHeader(entryHandle);
        if ((Archive.getUnsignedShort(this.bufs, this.offset + localHeader + 6L) & 0x41) != 0) {
            throw new IOException("Cannot read encrypted entries");
        }
        long offset = this.getDataOffset(localHeader);
        int method = this.getCompressionMethod(entryHandle);
        switch (method) {
            case 0: {
                return new ByteBufferInputStream(this.bufs, this.offset + offset, size);
            }
            case 8: {
                return new InflaterInputStream(new ByteBufferInputStream(this.bufs, this.offset + offset, size));
            }
        }
        throw new IOException("Unsupported compression scheme");
    }

    public Archive getNestedArchive(long entryHandle) throws IOException {
        ByteBuffer slice;
        long localHeader = this.getLocalHeader(entryHandle);
        if ((Archive.getUnsignedShort(this.bufs, this.offset + localHeader + 6L) & 0x41) != 0) {
            throw new IOException("Cannot read encrypted entries");
        }
        long offset = this.getDataOffset(localHeader);
        int method = this.getCompressionMethod(entryHandle);
        if (method != 0) {
            throw new IOException("Cannot open compressed nested archive");
        }
        long size = this.getUncompressedSize(entryHandle);
        if (size < Integer.MAX_VALUE && (slice = Archive.sliceOf(this.bufs, this.offset + offset, (int)size)) != null) {
            return Archive.open(slice);
        }
        return Archive.open(this.bufs, this.offset + offset, size);
    }

    public boolean isCompressed(long entryHandle) {
        return this.getCompressionMethod(entryHandle) != 0;
    }

    private int getCompressionMethod(long entryHandle) {
        return Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 10L);
    }

    private long getExtraRecord(long entryHandle, int headerId) {
        long extra = entryHandle + 46L + (long)Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 28L) + (long)Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 32L);
        int extraLen = Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 30L);
        for (int i = 0; i < extraLen; i += Archive.getUnsignedShort(this.bufs, this.offset + extra + (long)i + 2L)) {
            if (Archive.getUnsignedShort(this.bufs, this.offset + extra + (long)i) != headerId) continue;
            return extra + (long)i + 4L;
        }
        return -1L;
    }

    public long getUncompressedSize(long entryHandle) {
        long zip64;
        long size = Archive.getUnsignedInt(this.bufs, this.offset + entryHandle + 24L);
        if (size == -1L && (zip64 = this.getExtraRecord(entryHandle, 1)) != -1L) {
            size = Archive.getLong(this.bufs, this.offset + zip64 + 0L);
        }
        return size;
    }

    public long getCompressedSize(long entryHandle) {
        long zip64;
        long size = Archive.getUnsignedInt(this.bufs, this.offset + entryHandle + 20L);
        if (size == -1L && (zip64 = this.getExtraRecord(entryHandle, 1)) != -1L) {
            size = Archive.getLong(this.bufs, this.offset + zip64 + 8L);
        }
        return size;
    }

    public long getModifiedTime(long entryHandle) {
        long unixTime;
        long unix = this.getExtraRecord(entryHandle, 13);
        if (unix != -1L && (unixTime = Archive.getUnsignedInt(this.bufs, this.offset + unix + 4L)) != 0L) {
            return unixTime * 1000L;
        }
        return Archive.dosTimeStamp(Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 12L), Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 14L));
    }

    @Override
    public void close() {
    }

    private static long dosTimeStamp(int modTime, int modDate) {
        int year = 1980 + (modDate >> 9);
        int month = 1 + (modDate >> 5 & 0xF);
        int day = modDate & 0x1F;
        int hour = modTime >> 11;
        int minute = modTime >> 5 & 0x3F;
        int second = (modTime & 0x1F) << 1;
        return LocalDateTime.of(year, month, day, hour, minute, second).toInstant(ZoneOffset.UTC).toEpochMilli();
    }

    public boolean isDirectory(long entryHandle) {
        int madeBy = Archive.getUnsignedShort(this.bufs, this.offset + entryHandle + 4L);
        int extAttr = Archive.getInt(this.bufs, entryHandle + 38L);
        switch (madeBy) {
            case 3: {
                return (extAttr & 0xF000) == 16384;
            }
        }
        return (extAttr & 0x10) != 0;
    }

    static int bufIdx(long idx) {
        return (int)(idx >>> BUF_SHIFT);
    }

    static int bufOffs(long idx) {
        return (int)idx & 0x3FFFFFFF;
    }

    static byte getByte(ByteBuffer[] bufs, long idx) {
        return bufs[Archive.bufIdx(idx)].get(Archive.bufOffs(idx));
    }

    static int getUnsignedByte(ByteBuffer[] bufs, long idx) {
        return Archive.getByte(bufs, idx) & 0xFF;
    }

    static int getUnsignedByte(ByteBuffer buf, int idx) {
        return buf.get(idx) & 0xFF;
    }

    static short getShort(ByteBuffer[] bufs, long idx) {
        int bi = Archive.bufIdx(idx);
        return bi == Archive.bufIdx(idx + 1L) ? bufs[bi].getShort(Archive.bufOffs(idx)) : (short)(Archive.getUnsignedByte(bufs, idx) | Archive.getByte(bufs, idx + 1L) << 8);
    }

    static int getUnsignedShort(ByteBuffer[] bufs, long idx) {
        return Archive.getShort(bufs, idx) & 0xFFFF;
    }

    static int getMedium(ByteBuffer[] bufs, long idx) {
        return Archive.getUnsignedByte(bufs, idx) | Archive.getUnsignedShort(bufs, idx + 1L) << 8;
    }

    static long getUnsignedMedium(ByteBuffer[] bufs, long idx) {
        return Archive.getUnsignedByte(bufs, idx) | Archive.getUnsignedShort(bufs, idx + 1L) << 8;
    }

    static int getInt(ByteBuffer[] bufs, long idx) {
        int bi = Archive.bufIdx(idx);
        return bi == Archive.bufIdx(idx + 3L) ? bufs[bi].getInt(Archive.bufOffs(idx)) : Archive.getUnsignedShort(bufs, idx) | Archive.getShort(bufs, idx + 2L) << 16;
    }

    static long getUnsignedInt(ByteBuffer[] bufs, long idx) {
        return (long)Archive.getInt(bufs, idx) & 0xFFFFFFFFL;
    }

    static long getLong(ByteBuffer[] bufs, long idx) {
        int bi = Archive.bufIdx(idx);
        return bi == Archive.bufIdx(idx + 7L) ? bufs[bi].getLong(Archive.bufOffs(idx)) : Archive.getUnsignedInt(bufs, idx) | (long)Archive.getInt(bufs, idx + 4L) << 32;
    }

    static void readBytes(ByteBuffer[] bufs, long idx, byte[] dest, int off, int len) {
        while (len > 0) {
            int bi = Archive.bufIdx(idx);
            int bo = Archive.bufOffs(idx);
            ByteBuffer buf = bufs[bi].duplicate();
            buf.position(bo);
            int cnt = Math.min(len, buf.remaining());
            buf.get(dest, 0, cnt);
            len -= cnt;
            off += cnt;
            idx += (long)cnt;
        }
    }

    static byte[] getBytes(ByteBuffer[] bufs, long idx, int len) {
        byte[] bytes = new byte[len];
        Archive.readBytes(bufs, idx, bytes, 0, len);
        return bytes;
    }

    static ByteBuffer sliceOf(ByteBuffer[] bufs, long idx, int len) {
        int biEnd;
        if (len == 0) {
            return EMPTY_BUF;
        }
        int biStart = Archive.bufIdx(idx);
        if (biStart == (biEnd = Archive.bufIdx(idx + (long)len - 1L))) {
            ByteBuffer buf = bufs[biStart].duplicate();
            buf.position(Archive.bufOffs(idx));
            buf.limit(buf.position() + len);
            return buf.slice();
        }
        return null;
    }

    static ByteBuffer bufferOf(ByteBuffer[] bufs, long idx, int len) {
        ByteBuffer buf = Archive.sliceOf(bufs, idx, len);
        if (buf == null) {
            buf = ByteBuffer.wrap(Archive.getBytes(bufs, idx, len));
        }
        return buf;
    }

    static long capacity(ByteBuffer[] bufs) {
        int lastIdx = bufs.length - 1;
        return (long)lastIdx * 0x40000000L + (long)bufs[lastIdx].capacity();
    }

    static final class Utf8 {
        private Utf8() {
        }

        static int getByteCount(int a) {
            if (a <= 127) {
                return 1;
            }
            if (a <= 191) {
                return 1;
            }
            if (a <= 223) {
                return 2;
            }
            if (a <= 239) {
                return 3;
            }
            if (a <= 247) {
                return 4;
            }
            return 1;
        }

        static int codePointAt(ByteBuffer[] bufs, long i) {
            int a = Archive.getUnsignedByte(bufs, i);
            if (a <= 127) {
                return a;
            }
            if (a <= 191) {
                return 65533;
            }
            int b = Archive.getUnsignedByte(bufs, i + 1L);
            if ((b & 0xC0) != 128) {
                return 65533;
            }
            if (a <= 223) {
                return (a & 0x1F) << 6 | b & 0x3F;
            }
            int c = Archive.getUnsignedByte(bufs, i + 2L);
            if ((c & 0xC0) != 128) {
                return 65533;
            }
            if (a <= 239) {
                return (a & 0xF) << 12 | (b & 0x3F) << 6 | c & 0x3F;
            }
            int d = Archive.getUnsignedByte(bufs, i + 3L);
            if ((d & 0xC0) != 128) {
                return 65533;
            }
            if (a <= 247) {
                return (a & 7) << 18 | (b & 0x3F) << 12 | (c & 0x3F) << 6 | d & 0x3F;
            }
            return 65533;
        }
    }

    static final class Cp437 {
        static final char[] codePoints = new char[]{'\u0000', '\u263a', '\u263b', '\u2665', '\u2666', '\u2663', '\u2660', '\u2022', '\u25d8', '\u25cb', '\u25d9', '\u2642', '\u2640', '\u266a', '\u266b', '\u263c', '\u25ba', '\u25c4', '\u2195', '\u203c', '\u00b6', '\u00a7', '\u25ac', '\u21a8', '\u2191', '\u2193', '\u2192', '\u2190', '\u221f', '\u2194', '\u25b2', '\u25bc', ' ', '!', '\"', '#', '$', '%', '&', '\'', '(', ')', '*', '+', ',', '-', '.', '/', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?', '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~', '\u2302', '\u00c7', '\u00fc', '\u00e9', '\u00e2', '\u00e4', '\u00e0', '\u00e5', '\u00e7', '\u00ea', '\u00eb', '\u00e8', '\u00ef', '\u00ee', '\u00ec', '\u00c4', '\u00c5', '\u00c9', '\u00e6', '\u00c6', '\u00f4', '\u00f6', '\u00f2', '\u00fb', '\u00f9', '\u00ff', '\u00d6', '\u00dc', '\u00a2', '\u00a3', '\u00a5', '\u20a7', '\u0192', '\u00e1', '\u00ed', '\u00f3', '\u00fa', '\u00f1', '\u00d1', '\u00aa', '\u00ba', '\u00bf', '\u2310', '\u00ac', '\u00bd', '\u00bc', '\u00a1', '\u00ab', '\u00bb', '\u2591', '\u2592', '\u2593', '\u2502', '\u2524', '\u2561', '\u2562', '\u2556', '\u2555', '\u2563', '\u2551', '\u2557', '\u255d', '\u255c', '\u255b', '\u2510', '\u2514', '\u2534', '\u252c', '\u251c', '\u2500', '\u253c', '\u255e', '\u255f', '\u255a', '\u2554', '\u2569', '\u2566', '\u2560', '\u2550', '\u256c', '\u2567', '\u2568', '\u2564', '\u2565', '\u2559', '\u2558', '\u2552', '\u2553', '\u256b', '\u256a', '\u2518', '\u250c', '\u2588', '\u2584', '\u258c', '\u2590', '\u2580', '\u03b1', '\u00df', '\u0393', '\u03c0', '\u03a3', '\u03c3', '\u00b5', '\u03c4', '\u03a6', '\u0398', '\u03a9', '\u03b4', '\u221e', '\u03c6', '\u03b5', '\u2229', '\u2261', '\u00b1', '\u2265', '\u2264', '\u2320', '\u2321', '\u00f7', '\u2248', '\u00b0', '\u2219', '\u00b7', '\u221a', '\u207f', '\u00b2', '\u25a0', '\u00a0'};

        private Cp437() {
        }

        static char charFor(int c) {
            return codePoints[c];
        }
    }
}

