package net.lightapi.exporter;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;
import com.networknt.config.Config;
import com.networknt.config.JsonMapper;
import com.networknt.kafka.common.AvroConverter;
import com.networknt.kafka.common.AvroDeserializer;
import com.networknt.kafka.common.KafkaConsumerConfig;
import com.networknt.utility.NioUtils;
import org.apache.avro.specific.SpecificRecord;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.*;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.*;
import java.util.stream.Collectors;

import static java.io.File.separator;

/**
 * A Cli to export event from Kafka.
 *
 * @author Steve Hu
 */
public class Cli {

    @Parameter(names={"--filename", "-f"}, required = false,
            description = "The filename to be exported.")
    String filename;

    @Parameter(names={"--types", "-t"}, required = false,
            description = "The types of service to be exported. Concat with comma if multiple types.")
    String types;

    @Parameter(names={"--start", "-s"}, required = false,
            description = "The start timestamp to be exported.")
    String start;

    @Parameter(names={"--end", "-e"}, required = false,
            description = "The end timestamp to be exported.")
    String end;

    @Parameter(names={"--help", "-h"}, help = true)
    private boolean help;

    private static final Class<?> keyDeserializer = org.apache.kafka.common.serialization.ByteArrayDeserializer.class;
    private static final Class<?> valueDeserializer = org.apache.kafka.common.serialization.ByteArrayDeserializer.class;

    public static void main(String ... argv) throws Exception {
        try {
            Cli cli = new Cli();
            JCommander jCommander = JCommander.newBuilder()
                    .addObject(cli)
                    .build();
            jCommander.parse(argv);
            cli.run(jCommander);
        } catch (ParameterException e) {
            System.out.println("Command line parameter error: " + e.getLocalizedMessage());
            e.usage();
        }
    }

    public void run(JCommander jCommander) throws Exception {
        if (help) {
            jCommander.usage();
            return;
        }
        System.out.println("filename = " + filename + " types = " + types + " start = " + start + " end = " + end);
        // convert types to string array
        String[] typesArray = StringUtils.split(types, ',');
        EventMatcher matcher = new EventMatcher(typesArray);

        KafkaConsumerConfig config = (KafkaConsumerConfig) Config.getInstance().getJsonObjectConfig(KafkaConsumerConfig.CONFIG_NAME, KafkaConsumerConfig.class);
        // need to use a new group.id every time we run this command line.
        Properties props = new Properties();
        props.putAll(config.getProperties()); // Load base properties from config file

        // --- Use specific Deserializers ---
        props.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG, keyDeserializer);
        props.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG, valueDeserializer);

        // --- Critical Consumer Configs for this task ---
        // Don't use a group ID if manually assigning partitions and seeking
        // props.remove(ConsumerConfig.GROUP_ID_CONFIG);
        // OR use a unique one if still needed for some broker features, but manage offsets manually
        props.put(ConsumerConfig.GROUP_ID_CONFIG, "export-group-" + System.currentTimeMillis());
        props.put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, "false"); // Must manage commits or seeks manually
        props.put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, "none"); // Don't reset automatically when seeking

        long startTimestampMs;
        long endTimestampMs;
        try {
            // Parse timestamps - assumes ZULU format like '2025-04-09T02:32:10.178Z'
            startTimestampMs = OffsetDateTime.parse(start).toInstant().toEpochMilli();
            endTimestampMs = OffsetDateTime.parse(end).toInstant().toEpochMilli();
        } catch (Exception e) {
            e.printStackTrace();
            return;
        }

        props.put(ConsumerConfig.GROUP_ID_CONFIG, "" + System.currentTimeMillis());

        KafkaConsumer < byte[], byte[] > consumer = null;
        try(BufferedWriter writer = new BufferedWriter(new FileWriter(filename))) {
            consumer = new KafkaConsumer<>(props);
            System.out.println("Kafka consumer created.");
            // 1. Get partitions for the topic
            List<PartitionInfo> partitionInfoList = consumer.partitionsFor(config.getTopic());
            if (partitionInfoList == null || partitionInfoList.isEmpty()) {
                System.out.println("No partitions found for topic: " + config.getTopic());
                return;
            }
            System.out.println("Partitions for topic " + config.getTopic() + " "  + partitionInfoList);
            List<TopicPartition> topicPartitions = partitionInfoList.stream()
                    .map(pi -> new TopicPartition(config.getTopic(), pi.partition()))
                    .collect(Collectors.toList());
            System.out.println("Topic partitions: " + topicPartitions);
            // 2. Find starting offsets for the start timestamp
            Map<TopicPartition, Long> timestampsToSearch = topicPartitions.stream()
                    .collect(Collectors.toMap(tp -> tp, tp -> startTimestampMs));
            Map<TopicPartition, OffsetAndTimestamp> startingOffsets = consumer.offsetsForTimes(timestampsToSearch);
            System.out.println("Starting offsets for timestamps: " + startingOffsets);

            // 3. Assign partitions and seek
            consumer.assign(topicPartitions);
            KafkaConsumer<byte[], byte[]> finalConsumer = consumer;
            System.out.println("Assigned partitions: " + topicPartitions);
            startingOffsets.forEach((tp, offsetAndTimestamp) -> {
                if (offsetAndTimestamp != null) {
                    System.out.println("Seeking partition " + tp.partition() + " offset " +  offsetAndTimestamp.offset());
                    finalConsumer.seek(tp, offsetAndTimestamp.offset());
                } else {
                    // No offset found >= startTs, seek to end for this partition
                    // as we won't find relevant records from before startTs
                    System.out.println("No offset found for partition " + tp.partition()  + " >= startTs " + start +  " seeking to end.");
                    finalConsumer.seekToEnd(Collections.singletonList(tp));
                }
            });

            // 4. Poll and filter records
            final int giveUp = 5; // Adjust timeout logic as needed
            int noRecordsCount = 0;
            boolean continuePolling = true;

            while (continuePolling) {
                ConsumerRecords<byte[], byte[]> consumerRecords = consumer.poll(Duration.ofMillis(1000));

                if (consumerRecords.isEmpty()) {
                    noRecordsCount++;
                    if (noRecordsCount >= giveUp) {
                        System.out.println("No more records found after {} polls, stopping. " + giveUp);
                        continuePolling = false;
                    }
                    continue; // Go to next poll
                }

                noRecordsCount = 0; // Reset counter if records are found

                for (TopicPartition partition : consumerRecords.partitions()) {
                    List<ConsumerRecord<byte[], byte[]>> recordsForPartition = consumerRecords.records(partition);
                    for (ConsumerRecord<byte[], byte[]> record : recordsForPartition) {
                        // Check timestamp against end timestamp
                        if (record.timestamp() <= endTimestampMs) {
                            // Append key (as hex?) and value (as string)
                            String keyStr = new String(record.key(), StandardCharsets.UTF_8);
                            String valueStr = new String(record.value(), StandardCharsets.UTF_8);
                            // the value is a json string. convert it to a map
                            Map<String, Object> map = JsonMapper.string2Map(valueStr);
                            // check if the type is in the types array
                            String type = (String)map.get("type");
                            if(!matcher.matchesEvent(type)) {
                                System.out.println("Type " + type + " is not in the types array. Skip this record.");
                                continue;
                            }
                            // check if the event is in the ignored events list
                            List<String> ignoredEvents = getIgnoredEvents();
                            if(ignoredEvents.contains(type)) {
                                System.out.println("Type " + type + " is in the ignored events list. Skip this record.");
                                continue;
                            }
                            String s = keyStr + " " + valueStr ;
                            try {
                                writer.write(s);
                                writer.newLine();
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                        } else {
                            // Record is past the end timestamp for this partition
                            // We could potentially stop polling this specific partition, but
                            // for simplicity, we just stop processing records from it in this batch
                            System.out.printf("Record timestamp %s exceeds endTs %s for partition %s, stopping processing for this partition in this poll.%n",
                                    record.timestamp(), endTimestampMs, partition.partition());
                            // A more complex logic could pause/unassign partitions past the end time
                            // For now, we rely on the giveUp counter to eventually stop overall polling
                            break; // Stop processing this partition for this poll() call
                        }
                    }
                }
                // Note: No commit needed as we are not using group management for consumption progress
            }

        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("Failed to create Kafka consumer. Please check your configuration.");
            return;
        } finally {
            if (consumer != null) {
                consumer.close();
                System.out.println("Kafka consumer closed.");
            }
        }
        System.out.println("All Portal Events have been exported successfully to " + filename + ". Have fun!!!");
    }

    private static class ExportConsumerRebalanceListener implements ConsumerRebalanceListener {
        @Override
        public void onPartitionsRevoked(Collection < TopicPartition > partitions) {
            System.out.println("Called onPartitionsRevoked with partitions:" + partitions);
        }

        @Override
        public void onPartitionsAssigned(Collection<TopicPartition> partitions) {
            System.out.println("Called onPartitionsAssigned with partitions:" + partitions);
        }
    }

    private List<String> getIgnoredEvents() {
        List<String> list = new ArrayList<>();
        list.add("AuthCodeCreatedEvent");
        list.add("AuthCodeDeletedEvent");
        list.add("AuthRefreshTokenDeletedEvent");
        list.add("AuthRefreshTokenCreatedEvent");
        return list;
    }
}
