
package net.lightapi.portal.deployment.command.handler;

import com.networknt.monad.Failure;
import com.networknt.monad.Result;
import com.networknt.monad.Success;
import com.networknt.rpc.router.ServiceHandler;
import com.networknt.utility.UuidUtil;
import io.undertow.server.HttpServerExchange;
import net.lightapi.portal.HybridQueryClient;
import net.lightapi.portal.PortalConstants;
import net.lightapi.portal.command.AbstractCommandHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;

/**
 * Update a deployment instance in the portal-view UI deployment menu.
*/
@ServiceHandler(id="lightapi.net/deployment/updateDeploymentInstance/0.1.0")
public class UpdateDeploymentInstance extends AbstractCommandHandler {
    private static final Logger logger = LoggerFactory.getLogger(UpdateDeploymentInstance.class);

    @Override
    protected String getCloudEventType() {
        return PortalConstants.DEPLOYMENT_INSTANCE_UPDATED_EVENT;
    }

    @Override
    public String getCloudEventAggregateType() {
        return PortalConstants.AGGREGATE_DEPLOYMENT_INSTANCE;
    }

    @Override
    public String getCloudEventAggregateId(Map<String, Object> map) {
        // the aggregate id is the deploymentId in the data section.
        return (String) map.get("deploymentInstanceId");
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected Result<Map<String, Object>> enrichInput(HttpServerExchange exchange, Map<String, Object> map) {
        // need to generate the serviceId per legId
        String serviceId = (String) map.get("serviceId");
        String legId = (String) map.get("legId");
        if(legId != null && !legId.isBlank()) {
            map.put("serviceId", serviceId + "-" + legId);
        }
        // find the unique pipelineId for the deployment instance.
        String hostId = (String) map.get("hostId");
        String instanceId = (String) map.get("instanceId");
        String systemEnv = (String) map.get("systemEnv");
        String runtimeEnv = (String) map.get("runtimeEnv");
        Result<String> pipelineResult = HybridQueryClient.getDeploymentInstancePipeline(exchange, hostId, instanceId, systemEnv, runtimeEnv);
        if(pipelineResult.isFailure()) {
            return Failure.of(pipelineResult.getError());
        }
        String pipelineId = pipelineResult.getResult();
        map.put("pipelineId", pipelineId);
        return Success.of(map);
    }
}
