
package net.lightapi.portal.config.command.handler;

import com.networknt.config.JsonMapper;
import com.networknt.monad.Failure;
import com.networknt.monad.Result;
import com.networknt.monad.Success;
import com.networknt.rpc.router.ServiceHandler;
import io.undertow.server.HttpServerExchange;
import net.lightapi.portal.HybridQueryClient;
import net.lightapi.portal.PortalConstants;
import net.lightapi.portal.command.AbstractCommandHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Map;

/**
 * create a new config product version
*/
@ServiceHandler(id="lightapi.net/config/createConfigProductVersion/0.1.0")
public class CreateConfigProductVersion extends AbstractCommandHandler {
    private static final Logger logger = LoggerFactory.getLogger(CreateConfigProductVersion.class);
    private static final Base64.Encoder encoder = Base64.getMimeEncoder();

    @Override
    protected String getCloudEventType() {
        return PortalConstants.CONFIG_PRODUCT_VERSION_CREATED_EVENT;
    }

    @Override
    public String getCloudEventAggregateType() {
        return PortalConstants.AGGREGATE_CONFIG_PROPERTY;
    }

    @Override
    public String getCloudEventAggregateId(Map<String, Object> map) {
        // the aggregate id is the propertyId in the data section.
        return (String) map.get("propertyId");
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected Result<Map<String, Object>> enrichInput(HttpServerExchange exchange, Map<String, Object> map) {
        // get the property type to decide if we need to encode the value with base64.
        String configId = (String)map.get("configId");
        String propertyId = (String)map.get("propertyId");

        Result<String> configPropertyResult = HybridQueryClient.getConfigPropertyByPropertyId(exchange, configId, propertyId);
        if(configPropertyResult.isFailure()) {
            return Failure.of(configPropertyResult.getError());
        }
        String propertyType = JsonMapper.string2Map(configPropertyResult.getResult()).get("propertyType").toString();
        if(propertyType.equals("Cert") || propertyType.equals("File")) {
            // need to encode the value with base64
            String propertyValue = (String)map.remove("propertyValue");
            String encodedData = encoder.encodeToString(propertyValue.getBytes(StandardCharsets.UTF_8));
            map.put("propertyValue", encodedData);
        }
        return Success.of(map);
    }

}
