package net.lenni0451.mcstructs_bedrock.nbt.io.impl;

import net.lenni0451.mcstructs.nbt.INbtTag;
import net.lenni0451.mcstructs.nbt.NbtType;
import net.lenni0451.mcstructs.nbt.io.NbtHeader;
import net.lenni0451.mcstructs.nbt.io.types.INbtWriter;
import net.lenni0451.mcstructs.nbt.tags.*;

import javax.annotation.ParametersAreNonnullByDefault;
import java.io.DataOutput;
import java.io.IOException;
import java.util.Map;

@ParametersAreNonnullByDefault
public class BedrockNbtWriter implements INbtWriter {

    @Override
    public void writeType(DataOutput out, NbtType type) throws IOException {
        out.writeByte(type.getId());
    }

    @Override
    public void writeHeader(DataOutput out, NbtHeader header) throws IOException {
        if (header.isEnd()) {
            this.writeType(out, NbtType.END);
        } else {
            this.writeType(out, header.getType());
            BedrockWriteTypes.writeString(out, header.getName());
        }
    }

    @Override
    public void writeByte(DataOutput out, ByteTag tag) throws IOException {
        out.writeByte(tag.getValue());
    }

    @Override
    public void writeShort(DataOutput out, ShortTag tag) throws IOException {
        out.writeShort(tag.getValue());
    }

    @Override
    public void writeInt(DataOutput out, IntTag value) throws IOException {
        BedrockWriteTypes.writeVarInt(out, value.getValue());
    }

    @Override
    public void writeLong(DataOutput out, LongTag value) throws IOException {
        BedrockWriteTypes.writeVarLong(out, value.getValue());
    }

    @Override
    public void writeFloat(DataOutput out, FloatTag value) throws IOException {
        out.writeFloat(value.getValue());
    }

    @Override
    public void writeDouble(DataOutput out, DoubleTag value) throws IOException {
        out.writeDouble(value.getValue());
    }

    @Override
    public void writeByteArray(DataOutput out, ByteArrayTag value) throws IOException {
        BedrockWriteTypes.writeVarInt(out, value.getLength());
        out.write(value.getValue());
    }

    @Override
    public void writeString(DataOutput out, StringTag value) throws IOException {
        BedrockWriteTypes.writeString(out, value.getValue());
    }

    @Override
    public void writeList(DataOutput out, ListTag<?> value) throws IOException {
        if (value.getType() == null || value.isEmpty()) out.writeByte(NbtType.END.getId());
        else out.writeByte(value.getType().getId());
        BedrockWriteTypes.writeVarInt(out, value.size());
        for (INbtTag tag : value.getValue()) this.write(out, tag);
    }

    @Override
    public void writeCompound(DataOutput out, CompoundTag value) throws IOException {
        for (Map.Entry<String, INbtTag> entry : value.getValue().entrySet()) {
            this.writeHeader(out, new NbtHeader(entry.getValue().getNbtType(), entry.getKey()));
            this.write(out, entry.getValue());
        }
        this.writeHeader(out, NbtHeader.END);
    }

    @Override
    public void writeIntArray(DataOutput out, IntArrayTag value) throws IOException {
        BedrockWriteTypes.writeVarInt(out, value.getLength());
        for (int i : value.getValue()) BedrockWriteTypes.writeVarInt(out, i);
    }

    @Override
    public void writeLongArray(DataOutput out, LongArrayTag value) {
        throw new UnsupportedOperationException("LongArrayTag is not supported in Bedrock Nbt");
    }

}
