package net.lenni0451.commons.math.shapes.triangle;

import lombok.EqualsAndHashCode;
import lombok.ToString;

@ToString
@EqualsAndHashCode
public class TriangleF {

    private final float x1;
    private final float y1;
    private final float x2;
    private final float y2;
    private final float x3;
    private final float y3;

    public TriangleF(final float x1, final float y1, final float x2, final float y2, final float x3, final float y3) {
        this.x1 = x1;
        this.y1 = y1;
        this.x2 = x2;
        this.y2 = y2;
        this.x3 = x3;
        this.y3 = y3;
    }

    /**
     * @return The x coordinate of the first point
     */
    public float getX1() {
        return this.x1;
    }

    /**
     * @return The y coordinate of the first point
     */
    public float getY1() {
        return this.y1;
    }

    /**
     * @return The x coordinate of the second point
     */
    public float getX2() {
        return this.x2;
    }

    /**
     * @return The y coordinate of the second point
     */
    public float getY2() {
        return this.y2;
    }

    /**
     * @return The x coordinate of the third point
     */
    public float getX3() {
        return this.x3;
    }

    /**
     * @return The y coordinate of the third point
     */
    public float getY3() {
        return this.y3;
    }

    /**
     * Translate the triangle by the given values.
     *
     * @param x The x value to translate
     * @param y The y value to translate
     * @return The translated triangle
     */
    public TriangleF translate(final float x, final float y) {
        return new TriangleF(this.x1 + x, this.y1 + y, this.x2 + x, this.y2 + y, this.x3 + x, this.y3 + y);
    }

    /**
     * @return The area of the triangle
     */
    public double getArea() {
        return Math.abs((this.x1 * (this.y2 - this.y3) + this.x2 * (this.y3 - this.y1) + this.x3 * (this.y1 - this.y2)) / 2D);
    }

    /**
     * @return The perimeter of the triangle
     */
    public double getPerimeter() {
        double side1 = Math.hypot(this.x2 - this.x1, this.y2 - this.y1);
        double side2 = Math.hypot(this.x3 - this.x2, this.y3 - this.y2);
        double side3 = Math.hypot(this.x1 - this.x3, this.y1 - this.y3);
        return side1 + side2 + side3;
    }

    /**
     * @return The centroid of the triangle as an array [x, y]
     */
    public float[] getCentroid() {
        float centroidX = (this.x1 + this.x2 + this.x3) / 3;
        float centroidY = (this.y1 + this.y2 + this.y3) / 3;
        return new float[]{centroidX, centroidY};
    }

    /**
     * Check if a point is inside the triangle.
     *
     * @param px The x coordinate of the point
     * @param py The y coordinate of the point
     * @return If the point is inside the triangle
     */
    public boolean isPointInside(final float px, final float py) {
        double areaOrig = this.getArea();
        double area1 = Math.abs((px * (this.y2 - this.y3) + this.x2 * (this.y3 - py) + this.x3 * (py - this.y2)) / 2D);
        double area2 = Math.abs((this.x1 * (py - this.y3) + px * (this.y3 - this.y1) + this.x3 * (this.y1 - py)) / 2D);
        double area3 = Math.abs((this.x1 * (this.y2 - py) + this.x2 * (py - this.y1) + px * (this.y1 - this.y2)) / 2D);
        return (area1 + area2 + area3) == areaOrig;
    }

    /**
     * @return If the triangle is equilateral
     */
    public boolean isEquilateral() {
        double side1 = Math.hypot(this.x2 - this.x1, this.y2 - this.y1);
        double side2 = Math.hypot(this.x3 - this.x2, this.y3 - this.y2);
        double side3 = Math.hypot(this.x1 - this.x3, this.y1 - this.y3);
        return side1 == side2 && side2 == side3;
    }

}
