package net.lenni0451.commons.math.shapes.rect;

import lombok.EqualsAndHashCode;
import lombok.ToString;

@ToString
@EqualsAndHashCode
public class RectD {

    public static final RectD ZERO = new RectD(0, 0, 0, 0);

    private final double x;
    private final double y;
    private final double width;
    private final double height;

    public RectD(final double width, final double height) {
        this(0, 0, width, height);
    }

    public RectD(final double x, final double y, final double width, final double height) {
        this.x = x;
        this.y = y;
        this.width = width;
        this.height = height;
    }

    /**
     * @return The x coordinate of the top left corner
     */
    public double getX() {
        return this.x;
    }

    /**
     * @return The y coordinate of the top left corner
     */
    public double getY() {
        return this.y;
    }

    /**
     * @return The width of the rectangle
     */
    public double getWidth() {
        return this.width;
    }

    /**
     * @return The height of the rectangle
     */
    public double getHeight() {
        return this.height;
    }

    /**
     * Translate the rectangle by the given values.
     *
     * @param x The x translation
     * @param y The y translation
     * @return The translated rectangle
     */
    public RectD translate(final double x, final double y) {
        return new RectD(this.x + x, this.y + y, this.width, this.height);
    }

    /**
     * @return The area of the rectangle
     */
    public double getArea() {
        return this.width * this.height;
    }

    /**
     * @return If the rectangle is a square
     */
    public boolean isSquare() {
        return this.width == this.height;
    }

    /**
     * Scale the rectangle by the same factor in both directions.
     *
     * @param scale The scale factor
     * @return The scaled rectangle
     */
    public RectD scale(final double scale) {
        return this.scale(scale, scale);
    }

    /**
     * Scale the rectangle by different factors in both directions.
     *
     * @param scaleX The scale factor for the x direction
     * @param scaleY The scale factor for the y direction
     * @return The scaled rectangle
     */
    public RectD scale(final double scaleX, final double scaleY) {
        return new RectD(this.width * scaleX, this.height * scaleY);
    }

    /**
     * Check if this rectangle intersects with another rectangle.
     *
     * @param rect The other rectangle
     * @return If the rectangles intersect
     */
    public boolean intersects(final RectD rect) {
        return this.x < rect.x + rect.width && this.x + this.width > rect.x && this.y < rect.y + rect.height && this.y + this.height > rect.y;
    }

    /**
     * Check if this rectangle contains another rectangle.
     *
     * @param rect The other rectangle
     * @return If this rectangle contains the other rectangle
     */
    public boolean contains(final RectD rect) {
        return this.x <= rect.x && this.x + this.width >= rect.x + rect.width && this.y <= rect.y && this.y + this.height >= rect.y + rect.height;
    }

    /**
     * Get the intersection of this rectangle with another rectangle.
     *
     * @param rect The other rectangle
     * @return The intersection rectangle
     */
    public RectD intersection(final RectD rect) {
        if (!this.intersects(rect)) return ZERO;
        double x = Math.max(this.x, rect.x);
        double y = Math.max(this.y, rect.y);
        double width = Math.min(this.x + this.width, rect.x + rect.width) - x;
        double height = Math.min(this.y + this.height, rect.y + rect.height) - y;
        return new RectD(x, y, width, height);
    }

}
