package net.lenni0451.commons.math.shapes.point;

import lombok.EqualsAndHashCode;
import lombok.ToString;

@ToString
@EqualsAndHashCode
public class Point3D {

    private final double x;
    private final double y;
    private final double z;

    public Point3D(final double x, final double y, final double z) {
        this.x = x;
        this.y = y;
        this.z = z;
    }

    /**
     * @return The x coordinate of the point.
     */
    public double getX() {
        return this.x;
    }

    /**
     * @return The y coordinate of the point.
     */
    public double getY() {
        return this.y;
    }

    /**
     * @return The z coordinate of the point.
     */
    public double getZ() {
        return this.z;
    }

    /**
     * Add the given value to the point.
     *
     * @param value The value to add
     * @return The new point with the added value
     */
    public Point3D add(final double value) {
        return this.add(value, value, value);
    }

    /**
     * Add the given values to the point.
     *
     * @param x The x value to add
     * @param y The y value to add
     * @param z The z value to add
     * @return The new point with the added values
     */
    public Point3D add(final double x, final double y, final double z) {
        return new Point3D(this.x + x, this.y + y, this.z + z);
    }

    /**
     * Add the values of the given point to this point.
     *
     * @param point The point to add
     * @return The new point with the added values
     */
    public Point3D add(final Point3D point) {
        return this.add(point.getX(), point.getY(), point.getZ());
    }

    /**
     * Subtract the given value from the point.
     *
     * @param value The value to subtract
     * @return The new point with the subtracted value
     */
    public Point3D subtract(final double value) {
        return this.subtract(value, value, value);
    }

    /**
     * Subtract the given values from the point.
     *
     * @param x The x value to subtract
     * @param y The y value to subtract
     * @param z The z value to subtract
     * @return The new point with the subtracted values
     */
    public Point3D subtract(final double x, final double y, final double z) {
        return new Point3D(this.x - x, this.y - y, this.z - z);
    }

    /**
     * Subtract the values of the given point from this point.
     *
     * @param point The point to subtract
     * @return The new point with the subtracted values
     */
    public Point3D subtract(final Point3D point) {
        return this.subtract(point.getX(), point.getY(), point.getZ());
    }

    /**
     * Multiply the point with the given value.
     *
     * @param value The value to multiply
     * @return The new point with the multiplied value
     */
    public Point3D multiply(final double value) {
        return this.multiply(value, value, value);
    }

    /**
     * Multiply the point with the given values.
     *
     * @param x The x value to multiply
     * @param y The y value to multiply
     * @param z The z value to multiply
     * @return The new point with the multiplied values
     */
    public Point3D multiply(final double x, final double y, final double z) {
        return new Point3D(this.x * x, this.y * y, this.z * z);
    }

    /**
     * Multiply the values of the given point with this point.
     *
     * @param point The point to multiply
     * @return The new point with the multiplied values
     */
    public Point3D multiply(final Point3D point) {
        return this.multiply(point.getX(), point.getY(), point.getZ());
    }

    /**
     * Divide the point by the given value.
     *
     * @param value The value to divide
     * @return The new point with the divided value
     */
    public Point3D divide(final double value) {
        return this.divide(value, value, value);
    }

    /**
     * Divide the point by the given values.
     *
     * @param x The x value to divide
     * @param y The y value to divide
     * @param z The z value to divide
     * @return The new point with the divided values
     */
    public Point3D divide(final double x, final double y, final double z) {
        return new Point3D(this.x / x, this.y / y, this.z / z);
    }

    /**
     * Divide this point by the values of the given point.
     *
     * @param point The point to divide
     * @return The new point with the divided values
     */
    public Point3D divide(final Point3D point) {
        return this.divide(point.getX(), point.getY(), point.getZ());
    }

    /**
     * Calculate the square distance between this point and the given point.
     *
     * @param x The x coordinate of the point
     * @param y The y coordinate of the point
     * @param z The z coordinate of the point
     * @return The distance between the two points
     */
    public double distanceSquared(final double x, final double y, final double z) {
        return Math.pow(this.x - x, 2) + Math.pow(this.y - y, 2) + Math.pow(this.z - z, 2);
    }

    /**
     * Calculate the square distance between this point and the given point.
     *
     * @param point The point to calculate the distance to
     * @return The distance between the two points
     */
    public double distanceSquared(final Point3D point) {
        return this.distanceSquared(point.getX(), point.getY(), point.getZ());
    }

    /**
     * Calculate the distance between this point and the given point.
     *
     * @param x The x coordinate of the point
     * @param y The y coordinate of the point
     * @param z The z coordinate of the point
     * @return The distance between the two points
     */
    public double distance(final double x, final double y, final double z) {
        return Math.sqrt(this.distanceSquared(x, y, z));
    }

    /**
     * Calculate the distance between this point and the given point.
     *
     * @param point The point to calculate the distance to
     * @return The distance between the two points
     */
    public double distance(final Point3D point) {
        return this.distance(point.getX(), point.getY(), point.getZ());
    }

}
