package net.lenni0451.commons;

import lombok.experimental.UtilityClass;

import javax.annotation.Nullable;
import java.util.List;
import java.util.Random;

@UtilityClass
public class RandomUtils {

    private static final Random RND = new Random();

    /**
     * Generate a random string with the given length.<br>
     * The charset is: {@code a-z, A-Z, 0-9, _}
     *
     * @param length The length of the string
     * @return The generated string
     */
    public static String randomString(final int length) {
        return randomString(length, "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_");
    }

    /**
     * Generate a random string with the given length and charset.
     *
     * @param length  The length of the string
     * @param charset The charset to use
     * @return The generated string
     */
    public static String randomString(final int length, final String charset) {
        char[] out = new char[length];
        for (int i = 0; i < length; i++) out[i] = charset.charAt(RND.nextInt(charset.length()));
        return new String(out);
    }

    /**
     * Randomize the case of the given string.
     *
     * @param s The string to randomize
     * @return The randomized string
     */
    public static String randomizeCase(final String s) {
        char[] chars = s.toCharArray();
        for (int i = 0; i < chars.length; i++) {
            if (RND.nextBoolean()) chars[i] = Character.toUpperCase(chars[i]);
            else chars[i] = Character.toLowerCase(chars[i]);
        }
        return new String(chars);
    }

    /**
     * Generate a random integer between the given min and max.
     *
     * @param min The minimum value
     * @param max The maximum value
     * @return The generated integer
     */
    public static int randomInt(final int min, final int max) {
        if (min == max) return min;
        int rMin;
        int rMax;
        if (min > max) {
            rMin = max;
            rMax = min;
        } else {
            rMin = min;
            rMax = max;
        }

        return RND.nextInt(rMax - rMin) + rMin;
    }

    /**
     * Generate a random long between the given min and max.
     *
     * @param min The minimum value
     * @param max The maximum value
     * @return The generated long
     */
    public static long randomLong(final long min, final long max) {
        if (min == max) return min;
        long rMin;
        long rMax;
        if (min > max) {
            rMin = max;
            rMax = min;
        } else {
            rMin = min;
            rMax = max;
        }

        return (long) (RND.nextDouble() * (rMax - rMin)) + rMin;
    }

    /**
     * Generate a random double between the given min and max.
     *
     * @param min The minimum value
     * @param max The maximum value
     * @return The generated double
     */
    public static double randomDouble(final double min, final double max) {
        if (min == max) return min;
        double rMin;
        double rMax;
        if (min > max) {
            rMin = max;
            rMax = min;
        } else {
            rMin = min;
            rMax = max;
        }

        return RND.nextDouble() * (rMax - rMin) + rMin;
    }

    /**
     * Generate random bytes with the given length.
     *
     * @param length The length of the bytes
     * @return The generated bytes
     */
    public static byte[] randomBytes(final int length) {
        byte[] bytes = new byte[length];
        RND.nextBytes(bytes);
        return bytes;
    }

    /**
     * Get a random element from the given array.
     *
     * @param array The array to get the element from
     * @param <T>   The type of the array
     * @return The random element or null if the array is empty
     */
    @Nullable
    public static <T> T randomElement(final T[] array) {
        if (array.length == 0) return null;
        if (array.length == 1) return array[0];
        return array[RND.nextInt(array.length)];
    }

    /**
     * Get a random element from the given list.
     *
     * @param list The list to get the element from
     * @param <T>  The type of the list
     * @return The random element or null if the list is empty
     */
    @Nullable
    public static <T> T randomElement(final List<T> list) {
        if (list.isEmpty()) return null;
        if (list.size() == 1) return list.get(0);
        return list.get(RND.nextInt(list.size()));
    }

}
