package net.leanix.dropkit.oauth;

import io.dropwizard.auth.AuthenticationException;
import io.dropwizard.auth.Authenticator;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.JwtException;
import java.util.Optional;
import net.leanix.dropkit.oauth.models.User;
import net.leanix.dropkit.oauth.token.OAuth2Token;
import net.leanix.dropkit.oauth.token.OAuth2TokenParserInterface;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Authenticator used to protect resources.
 *
 *
 */
public class OAuth2Authenticator<U extends User> implements Authenticator<String, U> {
    private final Logger logger = LoggerFactory.getLogger(OAuth2Authenticator.class);
    private final OAuth2TokenParserInterface parser;
    private final Class<U> userClass;

    public OAuth2Authenticator(OAuth2TokenParserInterface parser, Class<U> userClass) {
        this.parser = parser;
        this.userClass = userClass;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.yammer.dropwizard.auth.Authenticator#authenticate(java.lang.Object)
     */
    @Override
    public Optional<U> authenticate(String accessToken) throws AuthenticationException {

        if (accessToken == null) {
            logger.warn("No token provided");
            return Optional.empty();
        }

        try {
            OAuth2Token<U> token = this.parser.parse(accessToken, userClass);
            return Optional.ofNullable(token.getPrincipal());
        } catch (ExpiredJwtException ex) {
            logger.warn("Token expired");
            return Optional.empty();
        } catch (JwtException ex) {
            logger.warn("Unable to verify token: ", ex);
            return Optional.empty();
        } catch (Throwable ex) {
            throw new AuthenticationException("Unable to authenticate token", ex);
        }
    }
}
