package net.leanix.dropkit.oauth.jwks;

import com.auth0.jwk.GuavaCachedJwkProvider;
import com.auth0.jwk.Jwk;
import com.auth0.jwk.JwkException;
import com.auth0.jwk.JwkProvider;
import com.auth0.jwk.UrlJwkProvider;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwsHeader;
import io.jsonwebtoken.SigningKeyResolver;
import java.security.Key;
import java.util.concurrent.TimeUnit;
import net.leanix.dropkit.oauth.OAuth2ResourceServerConfig;
import net.leanix.dropkit.oauth.token.ConfigException;
import net.leanix.dropkit.oauth.token.OAuth2TokenConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@linkplain SigningKeyResolver} which resolves a public signing key via JWKS endpoints and provide the legacy deployed
 * public key if jwks endpoint is not available or requested key is not provided.
 */
public class SigningKeyResolverJWKS implements SigningKeyResolver {

    private final static Logger LOG = LoggerFactory.getLogger(SigningKeyResolverJWKS.class);

    private final JwkProvider provider;
    private final Key fallbackPublicKey;

    public SigningKeyResolverJWKS(OAuth2TokenConfig oauth2TokenConfig) {
        this(new GuavaCachedJwkProvider(new UrlJwkProvider(oauth2TokenConfig.getJwksUri()), 20, 30, TimeUnit.MINUTES),
            tryReadingPublicKeyFromConfiguration(oauth2TokenConfig));
    }

    /**
     * Constructor used for junit tests only.
     * 
     * @param jwkProvider
     * @param fallbackKey
     */
    SigningKeyResolverJWKS(JwkProvider jwkProvider, Key fallbackKey) {
        this.provider = jwkProvider;
        this.fallbackPublicKey = fallbackKey;
    }

    private static Key tryReadingPublicKeyFromConfiguration(OAuth2TokenConfig oauth2TokenConfig) {
        if (oauth2TokenConfig == null ||
            (oauth2TokenConfig instanceof OAuth2ResourceServerConfig
                && ((OAuth2ResourceServerConfig) oauth2TokenConfig).getPublicKeyPath() == null)) {
            return null;
        }

        try {
            return oauth2TokenConfig.getSigningKey();
        } catch (ConfigException e) {
            LOG.info("Unable to load JWT public key from 'oauth' configuration. Try JWKS url lookup.");
        }

        return null;
    }

    @Override
    public Key resolveSigningKey(@SuppressWarnings("rawtypes") JwsHeader header, String plaintext) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Key resolveSigningKey(@SuppressWarnings("rawtypes") JwsHeader header, Claims claims) {
        try {
            Jwk jwk = provider.get(header.getKeyId());
            if (jwk != null) {
                return jwk.getPublicKey();
            }

        } catch (JwkException e) {
            LOG.error("Can't resolve JWK for keyId: " + header.getKeyId(), e);
        }

        // provide fixed deployed public key as fallback if available
        LOG.debug("Unable to lookup JWK for keyId '{}'. Try legacy deployed public key.", header.getKeyId());
        return fallbackPublicKey;
    }

}
