package net.leanix.dropkit;

import com.fasterxml.jackson.databind.util.StdDateFormat;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import io.dropwizard.Configuration;
import io.dropwizard.configuration.EnvironmentVariableSubstitutor;
import io.dropwizard.configuration.SubstitutingSourceProvider;
import io.dropwizard.setup.Bootstrap;
import io.dropwizard.setup.Environment;
import java.util.EnumSet;
import javax.servlet.DispatcherType;
import javax.servlet.FilterRegistration;
import org.eclipse.jetty.servlets.CrossOriginFilter;

/**
 * Application initialisation helper.
 *
 */
public class AppHelper {

    /**
     * Inits the use of iso8501 time format.
     * @param bootstrap configuration
     */
    public void initializeISO8601(Bootstrap<? extends Configuration> bootstrap) {
        bootstrap.getObjectMapper().registerModule(new JodaModule());
        bootstrap.getObjectMapper().setDateFormat(new StdDateFormat()/* ISO8601DateFormat() is deprecated */);
    }

    public void initialize(Bootstrap<? extends Configuration> bootstrap) {
        // Enable variable substitution with environment variables
        bootstrap.setConfigurationSourceProvider(
            new SubstitutingSourceProvider(
                bootstrap.getConfigurationSourceProvider(),
                new EnvironmentVariableSubstitutor(false)));
        this.initializeISO8601(bootstrap);
    }

    /**
     * To be included in App.run() method
     * @param config Configuration
     * @param environment Environment
     */
    public void run(ConfigurationWithCORS config, Environment environment) {
        this.addCORSFilter(environment, config);
        this.registerExceptionMappers(environment);
    }

    /**
     * Registers mappers for invalid-entity and business logic exception.
     * @param environment Environment
     */
    public void registerExceptionMappers(Environment environment) {
        // exception mappers
        environment.jersey().register(new InvalidEntityMapper());
        environment.jersey().register(new BusinessLogicExceptionMapper());
    }

    /**
     * Enables cross-origin requests.
     *
     *
     * @param environment Environment
     * @param config Configuration
     */
    public void addCORSFilter(Environment environment, ConfigurationWithCORS config) {
        if (config == null || config.getCrossOriginConfig() == null || config.getCrossOriginConfig().getWhitelist() == null) {
            return; // By default/unconfigured we don't add a CORS filter, i.e., no cross-origin requests for any Origin
        }
        if (config.getCrossOriginConfig().getWhitelist().equalsIgnoreCase("null")) {
            // Setting allowed origin to "null" is not recommended [1]. Additionally due to the nature of dropkit parsing
            // the config file, you get different results setting 'whitelist: null' or 'whitelist': "null"'.
            // Thus if set to null or "null" we don't allow cross-origin at all.
            // [1] https://w3c.github.io/webappsec-cors-for-developers/#avoid-returning-access-control-allow-origin-null
            return;
        }
        FilterRegistration.Dynamic filter = environment.servlets().addFilter("CORS", CrossOriginFilter.class);
        filter.addMappingForUrlPatterns(EnumSet.allOf(DispatcherType.class), true, "/*");
        filter.setInitParameter(CrossOriginFilter.ALLOWED_HEADERS_PARAM,
            "accept,accept-encoding,accept-language,access-control-request-headers,access-control-request-method,api-key,authorization,cache-control,content-type,content-length,connection,host,pragma,referer,x-requested-with,origin,x-api-sync-mode,x-api-key,x-credentials");
        filter.setInitParameter(CrossOriginFilter.ALLOWED_METHODS_PARAM, "GET,PUT,POST,DELETE,OPTIONS");
        filter.setInitParameter(CrossOriginFilter.PREFLIGHT_MAX_AGE_PARAM, "5184000"); // 2 months
        filter.setInitParameter(CrossOriginFilter.ALLOW_CREDENTIALS_PARAM, "true");
        filter.setInitParameter(CrossOriginFilter.ALLOWED_ORIGINS_PARAM, config.getCrossOriginConfig().getWhitelist());
    }
}
