package net.leanix.dropkit.oauth.jwks;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwtParser;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SigningKeyResolver;
import java.util.Date;
import java.util.UUID;
import net.leanix.dropkit.oauth.models.User;
import net.leanix.dropkit.oauth.token.ConfigException;
import net.leanix.dropkit.oauth.token.OAuth2Token;
import net.leanix.dropkit.oauth.token.OAuth2TokenConfig;

public class OAuth2TokenParserJWKS<U extends User> {

    private final ObjectMapper mapper = new ObjectMapper();

    private final Class<U> userClass;
    private final JwtParser jwtParser;

    public OAuth2TokenParserJWKS(OAuth2TokenConfig oauth2TokenConfig, Class<U> userClass) throws ConfigException {
        this(oauth2TokenConfig, userClass, new SigningKeyResolverJWKS(oauth2TokenConfig));
    }

    public OAuth2TokenParserJWKS(OAuth2TokenConfig oauth2TokenConfig,
        Class<U> userClass,
        SigningKeyResolver signingKeyResolver) throws ConfigException
    {
        this.userClass = userClass;
        this.jwtParser = Jwts.parser().setSigningKeyResolver(signingKeyResolver);
    }

    /**
     * Parse a Jwt and return OAuth2Token
     */
    public OAuth2Token<U> parse(String accessToken) {
        Claims claims = jwtParser.parseClaimsJws(accessToken).getBody();

        U user = mapper.convertValue(claims.get("principal"), userClass);

        OAuth2Token<U> token = new OAuth2Token<>();
        token.setId(UUID.fromString(claims.getId()));
        token.setPrincipal(user);
        user.setAccessToken(accessToken);
        token.setExpiration(claims.getExpiration());

        if (claims.get("refresh_token") != null) {
            token.setRefreshToken(claims.get("refresh_token", String.class));
        }

        if (claims.get("refresh_expiration") != null) {
            token.setRefreshExpiration(claims.get("refresh_exp", Date.class));
        }

        return token;
    }

}
