package net.leanix.dropkit.oauth.jwks;

import io.dropwizard.auth.AuthenticationException;
import io.dropwizard.auth.Authenticator;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.JwtException;
import java.util.Optional;
import javax.inject.Inject;
import javax.inject.Singleton;
import net.leanix.dropkit.oauth.models.User;
import net.leanix.dropkit.oauth.token.OAuth2Token;
import net.leanix.dropkit.oauth.token.OAuth2TokenConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * {@linkplain Authenticator} used to protect resources via JWT oauth2 tokens.
 * 
 * This implementation ensures the signature inside the JWT is valid using public RSA keys for
 * validation. The applied public keys will be fetched from a JWKS endpoint (TODO hardcoded right now) 
 * and if configured from a local deployed public key.
 */
@Singleton
public class OAuth2AuthenticatorJWKS<U extends User> implements Authenticator<String, U> {

    private final static Logger LOG = LoggerFactory.getLogger(OAuth2AuthenticatorJWKS.class);

    private final OAuth2TokenParserJWKS<U> parser;

    @Inject
    public OAuth2AuthenticatorJWKS(OAuth2TokenConfig config, Class<U> userClass) {
        try {
            this.parser = new OAuth2TokenParserJWKS<>(config, userClass);
        } catch (Exception e) {
            throw new RuntimeException("internal error", e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.yammer.dropwizard.auth.Authenticator#authenticate(java.lang.Object)
     */
    @Override
    public Optional<U> authenticate(String accessToken) throws AuthenticationException {

        if (accessToken == null) {
            LOG.warn("No token provided");
            return Optional.empty();
        }

        try {
            OAuth2Token<U> token = this.parser.parse(accessToken);
            return Optional.ofNullable(token.getPrincipal());
        } catch (ExpiredJwtException ex) {
            LOG.warn("Token expired");
            return Optional.empty();
        } catch (JwtException ex) {
            LOG.warn("Unable to verify token");
            return Optional.empty();
        } catch (Throwable ex) {
            throw new AuthenticationException("Unable to authenticate token", ex);
        }
    }
}
