package net.leanix.dropkit.quartz;

import javax.inject.Inject;

import org.quartz.Job;
import org.quartz.JobBuilder;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleScheduleBuilder;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.quartz.impl.StdSchedulerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.inject.Singleton;

import io.dropwizard.lifecycle.Managed;

/**
 * This is a analog implemenation to <code>com.fiestacabin.dropwizard.quartz.ManagedScheduler</code> used to setup time base running jobs.
 * The difference (and advantage) to ManagedScheduler is here that the job class does not need the <code>@Scheduled</code> annotation and
 * can be configured by standard dropwizard configuration mechanism.
 * <p>
 * Usage:
 * </p>
 * 
 * <pre>
 * // register SimpleManagedJobScheduler scheduler to start/stop with the jetty server
 * environment.lifecycle().manage(injector.getInstance(SimpleManagedJobScheduler.class));
 * // add job class
 * injector.getInstance(SimpleManagedJobScheduler.class).addJob(MyJob.class,
 *         config.getJobsConfiguration().getIntervallMillis());
 * </pre>
 * 
 * @author ralfwehner
 *
 */
@Singleton
public class SimpleManagedJobScheduler implements Managed {

    private static final Logger LOG = LoggerFactory.getLogger(SimpleManagedJobScheduler.class);

    private final Scheduler scheduler;
    private final LxGuiceJobFactory jobFactory;

    @Inject
    public SimpleManagedJobScheduler(Scheduler scheduler, LxGuiceJobFactory jobFactory) throws SchedulerException {
        this.scheduler = new StdSchedulerFactory().getScheduler();
        this.jobFactory = jobFactory;
    }

    public static SimpleManagedJobScheduler getInstance() throws SchedulerException {
        return new SimpleManagedJobScheduler(new StdSchedulerFactory().getScheduler(), null);
    }

    /**
     * Add a job that is executed each n milliseconds.
     */
    public void addJob(Class<? extends Job> jobClass, long intervalInMilliseconds)
            throws SchedulerException {

        if (intervalInMilliseconds <= 0) {
            throw new IllegalArgumentException("Parameter intervalInMilliseconds must be greater than 0.");
        }

        JobDetail jobDetail = JobBuilder.newJob(jobClass)
                .build();

        Trigger trigger = TriggerBuilder
                .newTrigger()
                .withSchedule(
                        SimpleScheduleBuilder.simpleSchedule()
                                .withIntervalInMilliseconds(intervalInMilliseconds).repeatForever())
                .build();
        LOG.info("Adding new job {} with trigger {} to quartz scheduler.", jobDetail, trigger);
        scheduler.scheduleJob(jobDetail, trigger);
    }

    public Scheduler getScheduler() {
        return scheduler;
    }

    @Override
    public void start() throws Exception {
        LOG.info("Starting quartz scheduler");
        if (jobFactory != null) {
            scheduler.setJobFactory(jobFactory);
        }
        scheduler.start();
    }

    @Override
    public void stop() throws Exception {
        LOG.info("Stopping quartz scheduler");
        scheduler.shutdown(true);
    }
}
