package net.leanix.dropkit.oauth.models;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.annotations.ApiModelProperty;

import java.io.Serializable;
import java.util.UUID;


/**
 * This class is a Data Transfer Object (DTO) representing a permission for a user to access a workspace (Defaults to active contact).
 *
 * In most of the cases the user is representing a persisted MTM user and the ID of the permission equals the ID from
 * the corresponding MTM permission on the workspace. Also then the permission not only includes a workspace ID & name
 * but also the customer roles and access control entities (ACE).
 * The role of this permission should be one of ADMIN, MEMBER, VIEWER or CONTACT.
 *
 * In some cases the user may represent a SYSTEM user (e.g. one of our microservices). A permission for this user to
 * access a workspace is generated "on the fly" and is not persisted within MTM.
 * Therefor the permission will not have an ID, nor customer roles or access control entities.
 * The permission will only hold a workspace ID & name. The role of this permission should be one of
 * SYSTEM_READ, SYSTEM_WRITE or SYSTEM_AS_USER (although the other roles may still make sense).
 *
 * Only in case the permission has the role SYSTEM_AS_USER the property "asUser" should be set. It will then hold a
 * user representing on of the persisted MTM users. This user again might reference a permission for which
 * the "persisted MTM user" paragraph above applies.
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class Permission implements Serializable {

    private static final long serialVersionUID = 1L;
    
    private UUID id;
    private UUID workspaceId;
    private String workspaceName;
    private PermissionRole role;
    private String[] customerRoles;
    private String[] accessControlEntities;
    private PermissionStatus status = PermissionStatus.ACTIVE;
    private User asUser;

    public Permission() {
    }

    public void setId(UUID id) {
        this.id = id;
    }

    @ApiModelProperty(dataType = "string")
    public UUID getId() {
        return id;
    }

    @ApiModelProperty(dataType = "string")
    public UUID getWorkspaceId() {
        return workspaceId;
    }

    public void setWorkspaceId(UUID workspaceId) {
        this.workspaceId = workspaceId;
    }

    public String getWorkspaceName() {
        return workspaceName;
    }

    public void setWorkspaceName(String workspaceName) {
        this.workspaceName = workspaceName;
    }

    @ApiModelProperty(dataType = "string")
    public PermissionStatus getStatus() {
        return status;
    }

    public void setStatus(PermissionStatus status) {
        this.status = status;
    }

    @JsonIgnore
    public boolean isActive() {
        return PermissionStatus.ACTIVE.equals(status);
    }

    @JsonIgnore
    public boolean isActive(PermissionRole role) {
        return isActive() && hasRole(role);
    }

    @ApiModelProperty(dataType = "string")
    public PermissionRole getRole() {
        return role;
    }

    public void setRole(PermissionRole role) {
        this.role = role;
    }

    public String[] getCustomerRoles() {
        return customerRoles;
    }

    public void setCustomerRoles(String[] customerRoles) {
        this.customerRoles = customerRoles;
    }

    public String[] getAccessControlEntities() {
        return accessControlEntities;
    }

    public void setAccessControlEntities(String[] accessControlEntities) {
        this.accessControlEntities = accessControlEntities;
    }

    /**
     * Checks if the permission has the given role.
     *
     * @param role
     * @return
     */
    public boolean hasRole(PermissionRole role) {
        return this.role != null && this.role.equals(role);
    }

    public User getAsUser() {
        return asUser;
    }

    public void setAsUser(User asUser) {
        this.asUser = asUser;
    }

    @Override
    public String toString() {
        return "[Permission: Workspace " + workspaceId + ", role " + role + ", status " + status.toString() + "]";
    }
}
