package net.leanix.dropkit.oauth.token;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwtParser;
import net.leanix.dropkit.oauth.models.User;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URL;
import java.util.Date;
import java.util.UUID;

public class OAuth2TokenParserBase implements OAuth2TokenParserInterface {

    private static final Logger log = LoggerFactory.getLogger(OAuth2TokenParserBase.class);
    private final JwtParser jwtParser;

    public OAuth2TokenParserBase(JwtParser jwtParser) {
        this.jwtParser = jwtParser;
    }

    /**
     * Parse a Jwt and return OAuth2Token
     */
    public <U extends User> OAuth2Token<U> parse(String accessToken, Class<U> userClass) {
        Claims claims = jwtParser.parseClaimsJws(accessToken).getBody();

        ObjectMapper mapper = new ObjectMapper();
        U user = mapper.convertValue(claims.get("principal"), userClass);
        try {
            user.setSvcUrl(new URL(claims.getIssuer()));
        } catch (Throwable e) {
        }

        OAuth2Token<U> token = new OAuth2Token<>();
        if (claims.getId() != null) {
            token.setId(UUID.fromString(claims.getId()));
        } else {
            log.error("no jti claim in JWT! Why? This is safe, and should disappear once all old JWT have expired");
        }
        token.setPrincipal(user);
        user.setAccessToken(accessToken);
        token.setExpiration(claims.getExpiration());

        if (claims.get("refresh_token") != null) {
            token.setRefreshToken(claims.get("refresh_token", String.class));
        }

        if (claims.get("refresh_expiration") != null) {
            token.setRefreshExpiration(claims.get("refresh_exp", Date.class));
        }

        return token;
    }
}
