package net.leanix.dropkit.oauth;

import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.inject.Inject;
import com.google.inject.Singleton;

import io.dropwizard.auth.AuthenticationException;
import io.dropwizard.auth.Authenticator;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.JwtException;
import net.leanix.dropkit.oauth.models.User;
import net.leanix.dropkit.oauth.token.OAuth2Token;
import net.leanix.dropkit.oauth.token.OAuth2TokenConfig;
import net.leanix.dropkit.oauth.token.OAuth2TokenParser;

/**
 * Authenticator used to protect resources.
 *
 *
 */
@Singleton
public class OAuth2Authenticator implements Authenticator<String, User> {
    private final Logger logger = LoggerFactory.getLogger(OAuth2Authenticator.class);
    private final OAuth2TokenParser parser;
    private static final ObjectMapper mapper = new ObjectMapper();

    static {
        mapper.disableDefaultTyping();
    }

    /**
     * Constructor.
     *
     * @param config
     */
    @Inject
    public OAuth2Authenticator(OAuth2TokenConfig config) {
        try {
            this.parser = new OAuth2TokenParser(config);
        } catch (Exception e) {
            throw new RuntimeException("unhandleable exception", e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.yammer.dropwizard.auth.Authenticator#authenticate(java.lang.Object)
     */
    @Override
    public Optional<User> authenticate(String accessToken) throws AuthenticationException {

        if (accessToken == null)
        {
            logger.warn("No token provided");
            return Optional.empty();
        }

        try {
            OAuth2Token token = this.parser.parse(accessToken);
            return Optional.ofNullable((User) token.getPrincipal());
        } catch (ExpiredJwtException ex) {
            logger.warn("Token expired, token = " + accessToken.substring(0, 10) + "...");
            return Optional.empty();
        } catch (JwtException ex) {
            logger.warn("Unable to verify, token = " + accessToken.substring(0, 10) + "...", ex);
            return Optional.empty();
        } catch (Throwable ex) {
            throw new AuthenticationException("Unable to authenticate, token = " + accessToken.substring(0, 10) + "...", ex);
        }
    }
}
