package net.leanix.dropkit.amqp;

import com.rabbitmq.client.AMQP.BasicProperties;
import com.rabbitmq.client.Channel;
import com.rabbitmq.client.DefaultConsumer;
import com.rabbitmq.client.Envelope;
import com.rabbitmq.client.ShutdownSignalException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.charset.Charset;

/**
 * Designed to consume from a single queue (holds the result from basicConsume
 * as registeredConsumerTag).
 */
public abstract class QueueConsumer extends DefaultConsumer {

    private final Logger log = LoggerFactory.getLogger(QueueConsumer.class);
    private final String queueName;
    private final ConsumerRegistry registry;

    private String registeredConsumerTag;

    private volatile boolean unregistering = false;

    private volatile long lastUsed;

    private final Charset utf8 = Charset.forName("UTF-8");

    public QueueConsumer(String queueName, Channel channel, ConsumerRegistry registry) {
        super(channel);
        this.queueName = queueName;
        this.registry = registry;
        this.lastUsed = System.currentTimeMillis();
    }

    public String getQueueName() {
        return queueName;
    }

    public String getRegisteredConsumerTag() {
        return registeredConsumerTag;
    }

    public void setRegisteredConsumerTag(String consumerTag) {
        this.registeredConsumerTag = consumerTag;
    }

    public void setUnregistering() {
        unregistering = true;
    }

    public long getLastUsed() {
        return lastUsed;
    }

    @Override
    public void handleDelivery(
            String consumerTag,
            Envelope env,
            BasicProperties props,
            byte[] body
    ) throws IOException {
        // we receive a delivery from a single thread only (no concurrency)
        log.debug("received message {} from queue {}", props.getMessageId(), queueName);

        lastUsed = System.currentTimeMillis();

        String stringBody = new String(body, utf8);

        simpleHandle(stringBody);

        getChannel().basicAck(env.getDeliveryTag(), false);
    }

    public abstract void simpleHandle(String body) throws IOException;

    @Override
    public void handleConsumeOk(String consumerTag) {
        log.info("consumer started consuming for queue {}, consumerTag={}", queueName, consumerTag);
    }

    @Override
    public void handleCancelOk(String consumerTag) {
        log.info("consumer for queue {}, consumerTag={}, was regularly cancelled. unregister it.", queueName, consumerTag);
        if (!unregistering) {
            unregistering = true;
            registry.unregister(this);
        }
    }

    @Override
    public void handleCancel(String consumerTag) {
        log.info("consumer for queue {}, consumerTag={}, was cancelled, e.g. because queue was deleted. unregister it.", queueName, consumerTag);
        if (!unregistering) {
            unregistering = true;
            registry.unregister(this);
        }
    }

    @Override
    public void handleShutdownSignal(String consumerTag, ShutdownSignalException e) {
        log.info("{} for consumer for queue {}, consumerTag={}, was closed. unregister consumer.", e.isHardError() ? "connection" : "channel", queueName, consumerTag);
        if (!unregistering) {
            unregistering = true;
            registry.unregister(this);
        }
    }
}
