package net.leanix.dropkit.oauth;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Optional;
import com.google.inject.Inject;

import io.dropwizard.auth.AuthenticationException;
import io.dropwizard.auth.Authenticator;
import net.leanix.dropkit.oauth.models.Principal;
import net.leanix.dropkit.oauth.token.OAuth2Token;
import net.leanix.dropkit.oauth.token.OAuth2TokenConfig;
import net.leanix.dropkit.oauth.token.OAuth2TokenParser;

/**
 * Authenticator used to protect resources.
 *
 *
 */
public class OAuth2Authenticator implements Authenticator<String, Principal> {
    private final Logger logger = LoggerFactory.getLogger(OAuth2Authenticator.class);
    private final OAuth2TokenParser parser;
    private static final ObjectMapper mapper = new ObjectMapper();

    static {
        mapper.disableDefaultTyping();
    }

    /**
     * Constructor.
     *
     * @param config
     */
    @Inject
    public OAuth2Authenticator(OAuth2TokenConfig config) throws Exception {
        this.parser = new OAuth2TokenParser(config);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.yammer.dropwizard.auth.Authenticator#authenticate(java.lang.Object)
     */
    @Override
    public Optional<Principal> authenticate(String accessToken) throws AuthenticationException {
        try {
            OAuth2Token token = this.parser.parse(accessToken);
            return Optional.fromNullable(token.getPrincipal());
        } catch (Throwable ex) {
            logger.error("Unable to verify token " + accessToken.substring(0, 10) + "..., reason: " + ex.getMessage(), ex);
            throw new AuthenticationException("Unable to authenticate token.", ex);
        }
    }
}
