package net.leanix.dropkit.util;

import java.net.URI;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.leanix.dropkit.etcd.EtcdClient;
import net.leanix.dropkit.etcd.EtcdClientException;
import net.leanix.dropkit.etcd.EtcdResult;

/**
 * Helper class to detect if current instance is active or not.<br/>
 * 
 * The logic to detect if instance is active or inactive is based on the configuration in etcd under key
 * <code>/vhosts/local-svc.leanix.net/deploy_current</code> or <code>/vhosts/local-eam.leanix.net/deploy_current</code>. In oder to use this
 * class you have to specify:
 * 
 * <pre>
 * * ETCD_SERVER    : the etcd server, e.g. '192.168.59.103:4001'  
 * * CONTAINER_COLOR: 'green', 'blue' or 'white'
 * * VIRTUAL_HOST   : the virtual host name e.g. 'local-eam.leanix.net' (default: local-svc.leanix.net)
 * </pre>
 * 
 * @author ralfwehner
 *
 */
public class DeploymentUtil {

    static String ROOT_VHOSTS = "/vhosts/";
    public static final String VIRTUAL_HOST = "VIRTUAL_HOST";
    public static final String CONTAINER_COLOR = "CONTAINER_COLOR";
    public static final String ETCD_SERVER = "ETCD_SERVER";

    private static final Logger LOG = LoggerFactory.getLogger(DeploymentUtil.class);

    private final DeploymentColor containerColor;
    final URI etcdUri;

    public DeploymentUtil() {
        String color = getEnvironmentVariable(CONTAINER_COLOR);
        containerColor = DeploymentColor.fromString(color);
        LOG.debug("Detected assgigned deployment color '{}' to this container.", containerColor);
        String etcdServer = getEnvironmentVariable(ETCD_SERVER);
        if (StringUtils.isEmpty(etcdServer)) {
            LOG.info(
                    "No etcd server configuration detected. Is this container running in an on premise environment? ETCD configuration will be skipped.");
            etcdUri = null;
        } else {
            etcdUri = URI.create(String.format("http://%s", etcdServer));
        }
    }

    public DeploymentColor getContainerColor() {
        return containerColor;
    }

    public boolean isInstanceCurrentlyActive() {

        // If we can not detect the color of this container, maybe we are running in an on premise installation, this container returns NO.
        if (containerColor == null) {
            LOG.warn("Can not detect container color and return: NO - this container is not currently active.");
            return false;
        } else if (containerColor == DeploymentColor.WHITE) {
            return true;
        } else if (etcdUri == null) {
            if (containerColor != DeploymentColor.WHITE) {
                LOG.warn(
                        "Althought the container color '{}' is specified, no environment variable '{}' exists to specify the etcd configuration.",
                        containerColor, ETCD_SERVER);
            }
            return true;
        }

        // setup key to look for in etcd
        String virtualHost = getEnvironmentVariable(VIRTUAL_HOST);
        if (StringUtils.isEmpty(virtualHost)) {
            virtualHost = "local-svc.leanix.net";
            LOG.warn("Environment variable '{}' not found. Using '{}' as default.", VIRTUAL_HOST, virtualHost);
        }
        String key = ROOT_VHOSTS + virtualHost.trim() + "/deploy_current";

        // read from etcd
        EtcdClient client = new EtcdClient(etcdUri);

        try {
            EtcdResult result = client.get(key);
            if (result != null && !result.isError()) {
                String currentColor = result.getNode().getValue();

                LOG.debug("Read current deployment color '{}' from etcd. This container has color {}. (URL: {}, key: '{}')",
                        currentColor, containerColor, etcdUri, key);
                return (containerColor.toString().equalsIgnoreCase(currentColor));
            }
        } catch (EtcdClientException e) {
            LOG.warn("Can not connect to etcd server! (URL: " + etcdUri + ")", e);
        } catch (Exception e) {
            LOG.warn("Internal error: Can not connect to etcd server! (URL: " + etcdUri + ")", e);
        }

        // etcd configuration is required, but we can not access it.
        return false;
    }

    private String getEnvironmentVariable(String varName) {
        String env = System.getenv(varName);
        if (env == null) {
            LOG.info("Environment varible '{}' not found in Sytem.env(). Try to read from system.properties.", varName);
            return System.getProperty(varName);
        }
        return env;
    }
}
