package net.leanix.dropkit.amqp;

import java.io.IOException;

import com.rabbitmq.client.AMQP.BasicProperties;
import com.rabbitmq.client.ConnectionFactory;
import com.rabbitmq.client.Consumer;
import com.rabbitmq.client.Envelope;
import com.rabbitmq.client.ShutdownSignalException;
import org.joda.time.DateTime;
import org.slf4j.LoggerFactory;

/**
 * The endpoint that consumes messages off of the queue. Happens to be runnable.
 *
 * @author syntx
 * @link http://java.dzone.com/articles/getting-started-rabbitmq-java
 */
public class QueueConsumer extends Endpoint implements Runnable, Consumer {

    private final DeliveryHandler handler;
    private final String bindingKey;
    private DateTime lastUsage;
    
    public QueueConsumer(AMQPConfiguration config, ConnectionFactory connectionFactory, DeliveryHandler handler) throws IOException {
        super(config, connectionFactory);
        this.handler = handler;
        this.bindingKey = null;
        this.lastUsage = new DateTime(Integer.MIN_VALUE);
    }
    
    public QueueConsumer(AMQPConfiguration config, ConnectionFactory factory, DeliveryHandler handler, String bindingKey) throws IOException {
        super(config, factory);
        this.handler = handler;
        this.bindingKey = bindingKey;
        this.lastUsage = new DateTime(Integer.MIN_VALUE);

        channel.queueBind(config.getQueue(), config.getExchange(), bindingKey);

        LoggerFactory.getLogger(QueueConsumer.class).info("New QueueConsumer with bindingKey: " + bindingKey + " started.");
    }
    
    @Override
    public void run() {
        try {
            LoggerFactory.getLogger(QueueConsumer.class).info("QueueConsumer with bindingKey: " + bindingKey + " starts to consume in thread " + Thread.currentThread().getName());
            //start consuming messages. Auto acknowledge messages.
            channel.basicConsume(this.configuration.getQueue(), true, this);
        } catch (IOException e) {
            LoggerFactory.getLogger(QueueConsumer.class).error("QueueConsumer error: ", e);
        }
    }

    public void registerWithChannel() throws IOException {
        //start consuming messages. Auto acknowledge messages.
        channel.basicConsume(this.configuration.getQueue(), true, this);
    }

    /**
     * Called when consumer is registered.
     *
     * @param consumerTag
     */
    @Override
    public void handleConsumeOk(String consumerTag) {
        System.out.println("Consumer " + consumerTag + " registered");
    }

    /**
     * Called when new message is available.
     *
     * @param consumerTag
     * @param env
     * @param props
     * @param body
     * @throws java.io.IOException
     */
    @Override
    public void handleDelivery(
            String consumerTag,
            Envelope env,
            BasicProperties props,
            byte[] body
    ) throws IOException {
        handler.handleDelivery(props, body);
    }

    @Override
    public void handleCancel(String consumerTag) {
    }

    @Override
    public void handleCancelOk(String consumerTag) {
    }

    @Override
    public void handleRecoverOk(String consumerTag) {
    }

    @Override
    public void handleShutdownSignal(String consumerTag, ShutdownSignalException arg1) {
    }
    
    public String getBindingKey() {
        return bindingKey;
    }
    
    public void setLastUsage(DateTime lastUsage) {
        this.lastUsage = lastUsage;
    }

    public DateTime getLastUsage() {
        return lastUsage;
    }
}
