package net.leanix.dropkit.oauth;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.core.util.Base64;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import net.leanix.dropkit.api.ClientFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides an access token for two legged oauth.
 *
 *
 */
public class ClientCredentialAccessTokenFactory {

    private final OAuth2ClientConfig config;
    private final String authorization;
    private final com.sun.jersey.api.client.Client apiClient;

    private final Logger logger;

    private AccessTokenResponse accessTokenResponse;

    private final long expireGraceTime = 10;

    /**
     * Factory method.
     *
     * @param config
     * @return
     */
    public static ClientCredentialAccessTokenFactory create(OAuth2ClientConfig config) {
        com.sun.jersey.api.client.Client apiClient = ClientFactory.createJerseyClientWithJacksonSerializer();
        Logger logger = LoggerFactory.getLogger(ClientCredentialAccessTokenFactory.class);
        return new ClientCredentialAccessTokenFactory(config, apiClient, logger);
    }

    public ClientCredentialAccessTokenFactory(
            OAuth2ClientConfig config,
            Client apiClient,
            Logger logger
    ) {
        authorization = "Basic ".concat(
                new String(Base64.encode(config.getClientId().concat(":").concat(config.getClientSecret()).getBytes()))
        );
        this.config = config;
        this.apiClient = apiClient;
        this.logger = logger;
    }

    public String getAccessToken() throws FlowException {
        if (getAccessTokenResponse() == null || !verifyToken()) {
            fetchToken();
        }

        return getAccessTokenResponse().getAccessToken();
    }

    protected AccessTokenResponse getAccessTokenResponse() {
        return this.accessTokenResponse;
    }

    private boolean verifyToken() throws FlowException {
        VerifyTokenResponse verificationResponse = null;
        try {
            verificationResponse = apiClient
                    .resource(String.format(config.getVerificationUrl().concat("?access_token=%s"), accessTokenResponse.getAccessToken()))
                    .accept(MediaType.APPLICATION_JSON)
                    .header(HttpHeaders.AUTHORIZATION, authorization)
                    .get(VerifyTokenResponse.class);
        } catch (ClientHandlerException | UniformInterfaceException ex) {
            logger.info("Failed to verify access_token " + config.getTokenUrl() + " using " + authorization + " against " + config.getVerificationUrl(), ex);
            return false;
        }

        if (verificationResponse == null) {
            throwFlowException("Verification response null against " + config.getVerificationUrl());
        }

        if ((verificationResponse.getError() == null || verificationResponse.getError().isEmpty()) && verificationResponse.getExpiresIn() > expireGraceTime) {
            logger.debug("Successful access token " + accessTokenResponse.getAccessToken() + " verification, expires in " + verificationResponse.getExpiresIn() + " sec");
            return true;
        }

        logger.debug("Unable to verify access token " + accessTokenResponse.getAccessToken() + ", error = " + verificationResponse.getError() + " expire = " + verificationResponse.getExpiresIn());
        return false;
    }

    private void fetchToken() throws FlowException {
        try {
            accessTokenResponse = apiClient
                    .resource(config.getTokenUrl().concat("?grant_type=client_credentials"))
                    .accept(MediaType.APPLICATION_JSON)
                    .header(HttpHeaders.AUTHORIZATION, authorization)
                    .header(HttpHeaders.ACCEPT, "application/json")
                    .post(AccessTokenResponse.class);
        } catch (ClientHandlerException | UniformInterfaceException ex) {
            throwFlowException("Failed to retrieve a new oauth token from " + config.getTokenUrl() + " using " + authorization + " against " + config.getTokenUrl(), ex);
        }

        logger.info("Fetched a new token: " + accessTokenResponse.getAccessToken());
    }

    private void throwFlowException(String message, Exception ex) throws FlowException {
        logger.error(message, ex);
        throw new FlowException(message);
    }

    private void throwFlowException(String message) throws FlowException {
        logger.error(message);
        throw new FlowException(message);
    }
}
