package net.leanix.dropkit.api;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.cache.Cache;
import com.sun.jersey.api.client.WebResource;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Api client.
 *
 *
 */
public class Client {

    private final String baseUrl;
    private final com.sun.jersey.api.client.Client apiClient;
    private Cache cache;

    /**
     * Constructor
     *
     * @param baseUrl the api base url
     * @param apiClient a preconfigured client (except auth)
     */
    public Client(
            String baseUrl,
            com.sun.jersey.api.client.Client apiClient
    ) {
        this.baseUrl = baseUrl;
        this.apiClient = apiClient;
    }

    /**
     * Constructor that enables caching.
     *
     * @param baseUrl the api base url
     * @param apiClient a preconfigured client (except auth)
     * @param cache cache implementation
     */
    public Client(
            String baseUrl,
            com.sun.jersey.api.client.Client apiClient,
            Cache cache
    ) {
        this.baseUrl = baseUrl;
        this.apiClient = apiClient;
        this.cache = cache;
    }

    public void setCache(Cache cache) {
        this.cache = cache;
    }

    public WebResource resource(String path) {
        return apiClient.resource(getUrl(path));
    }

    public Object getCachedGet(String url) {
        if (cache == null) {
            return null;
        }

        return cache.getIfPresent(url);
    }

    public void setCachedGet(String url, Object o) {
        if (cache == null) {
            return;
        }

        cache.put(url, o);
    }

    public Object getCachedPost(String url, Object o) {
        if (cache == null) {
            return null;
        }

        try {
            return cache.getIfPresent(url + getObjectStringHash(o));
        } catch (JsonProcessingException ex) {
            Logger.getLogger(Client.class.getName()).log(Level.WARNING, null, ex);
            return null;
        }
    }

    public void setCachedPost(String url, Object o, Object result) {
        if (cache == null) {
            return;
        }

        try {
            String key = url + getObjectStringHash(o);
            cache.put(key, result);
        } catch (JsonProcessingException ex) {
            Logger.getLogger(Client.class.getName()).log(Level.WARNING, null, ex);
        }
    }

    /**
     * Returns the base url plus the path.
     *
     * @param path
     * @return
     */
    public String getUrl(String path) {
        return baseUrl + path;
    }

    /**
     * Serializes the object to string and computes the hash code.
     *
     * In the case of Strings the hash code can be used as checksum.
     *
     * @param o
     * @return
     * @throws JsonProcessingException
     */
    private int getObjectStringHash(Object o) throws JsonProcessingException {
        ObjectMapper mapper = new ObjectMapper();
        return mapper.writeValueAsString(o).hashCode();
    }
}
